import logging
import re
import sform

from datetime import date
from functools import partial

from django.core.exceptions import ValidationError
from django.core.validators import URLValidator
from django.utils.datastructures import OrderedSet

from intranet.femida.src.api.candidates.forms import (
    CandidateContactForm,
    CandidateEducationForm,
    CandidateJobForm,
)
from intranet.femida.src.api.core.forms import UserSuggestField
from intranet.femida.src.api.core.validators import validate_phone
from intranet.femida.src.beamery.tasks import log_invalid_data_task, log_fatally_invalid_data_task
from intranet.femida.src.candidates.bulk_upload.choices import CANDIDATE_UPLOAD_MODES
from intranet.femida.src.candidates.choices import (
    CANDIDATE_STATUSES,
    CANDIDATE_DEGREES,
    CONTACT_TYPES,
    CONTACT_SOURCES,
)
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.choices import GENDER_CHOICES
from intranet.femida.src.core.exceptions import SimpleValidationError


logger = logging.getLogger(__name__)


Required = partial(ValidationError, 'This field is required.', code='required')
INVALID = object()


def _reformat_contact(account_id, type=CONTACT_TYPES.other, is_main=False):
    return dict(
        account_id=account_id,
        type=type,
        is_main=is_main,
    )


def _parse_partial_date(date_str):
    """
    Парсит дату в формате YYYY[-MM[-DD]].
    Недостающие значения заменяются на 1.

    Если распарсить не удалось, возвращает None.

    Например:
    2012-10-11 -> date(2012, 10, 11)
    2014-04 -> date(2014, 4, 1)
    2018 -> date(2018, 1, 1)
    02.10.1983 -> None
    2018-13-01 -> None

    :param date_str: строка с датой формата YYYY[-MM[-DD]]
    :return: date|None
    """
    date_re = re.compile(
        r'^(?P<year>\d{4})(-(?P<month>\d{2})(-(?P<day>\d{2}))?)?$'
    )
    match = date_re.match(date_str.strip())
    if not match:
        return None

    kw = {k: int(v) for k, v in match.groupdict('1').items()}
    try:
        return date(**kw)
    except ValueError:
        return None


class SkipInvalidGridField(sform.GridField):
    """
    GridField, который не кидает ошибку, а пропускает невалидные значения
    """
    def clean(self, new_value, old_value, required, *args, **kwargs):
        cd = super().clean(new_value, old_value, required, *args, **kwargs)
        result = [i for i in cd if i is not INVALID]
        if required and not result:
            raise Required()
        return result


class SkipInvalidFieldMixin:
    """
    Миксин, который позволяет не кидать ошибку валидации,
    вместо этого он отдаёт специальный объект INVALID.
    Имеет смысл использовать только для полей, используемых внутри SkipInvalidGridField.
    """
    def clean(self, *args, **kwargs):
        try:
            return super().clean(*args, **kwargs)
        except ValidationError:
            return INVALID


class SkipInvalidCharField(SkipInvalidFieldMixin, sform.CharField):
    pass


class SkipInvalidFieldsetField(SkipInvalidFieldMixin, sform.FieldsetField):
    pass


ContactsField = partial(SkipInvalidGridField, SkipInvalidFieldsetField(CandidateContactForm))
EducationsField = partial(SkipInvalidGridField, SkipInvalidFieldsetField(CandidateEducationForm))
JobsField = partial(SkipInvalidGridField, SkipInvalidFieldsetField(CandidateJobForm))
TagsField = partial(SkipInvalidGridField, SkipInvalidCharField(max_length=255))


class CommaSeparatedField(sform.CharField):
    """
    Трансформирует строку со значениями
    через запятую в список значений.

    Пустые значения пропускаются.
    """
    def clean(self, *args, **kwargs):
        value = super().clean(*args, **kwargs)
        if not value:
            return []
        stripped_values_gen = (i.strip() for i in value.split(','))
        return [i for i in stripped_values_gen if i]


class CommaSeparatedMappingField(CommaSeparatedField):
    """
    Принимает строку с ключами через запятую
    и трансформирует это в список значений соответствующих ключам в словаре.
    Пустые значения пропускаются.

    Сам словарь ожидается в начальных данных формы
    initial['_mapping'][mapping_name]
    , где mapping_name - это параметр заданный при инициализации поля.

    По умолчанию, словарь пуст.
    """
    def __init__(self, mapping_name, strict=False, case_sensitive=True, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.mapping_name = mapping_name
        self.strict = strict

        # Если False, для всех значений ищем ключ в словаре в нижнем регистре.
        # Соответственно, в таком случае ожидается, что все ключи
        # соответствующего словаря тоже в нижнем регистре.
        self.case_sensitive = case_sensitive

    def clean(self, *args, **kwargs):
        values = super().clean(*args, **kwargs)
        initial = kwargs['base_initial']
        mapping = initial.get('_mapping', {}).get(self.mapping_name, {})

        result = OrderedSet()
        invalid_values = []
        for value in values:
            value = value if self.case_sensitive else value.lower()
            if value in mapping:
                result.add(mapping[value])
            else:
                invalid_values.append(value)

        if invalid_values and self.strict:
            raise ValidationError(
                message='',
                code='invalid_choice',
                params={
                    'values': invalid_values,
                },
            )

        return list(result)


class PartialDateField(sform.CharField):
    """
    Принимает дату в формате YYYY[-MM[-DD]].
    Недостающие значения заменяются на 1.
    """
    def clean(self, *args, **kwargs):
        value = super().clean(*args, **kwargs)
        if not value:
            return None
        parsed = _parse_partial_date(value)
        if parsed is None:
            raise ValidationError(
                message='',
                code='invalid',
            )
        return parsed


class NullYesField(sform.NullBooleanField):
    def __init__(self, *args, **kwargs):
        kwargs['default'] = kwargs.get('default', '')
        super().__init__(*args, **kwargs)

    def extract_new_value(self, data, name):
        val = super().extract_new_value(data, name).lower().strip()
        if val in ['yes', 'да']:
            return True
        elif val in ['no', 'нет']:
            return False
        return None


class CandidateBulkUploadBaseForm(sform.SForm):
    """
    Базовая форма для массовой загрузки кандидатов
    """
    original = sform.SuggestField(
        queryset=Candidate.unsafe.all(),
        label_fields=['pk'],
    )
    first_name = sform.CharField(max_length=255)
    middle_name = sform.CharField(max_length=255)
    last_name = sform.CharField(max_length=255)
    birthday = sform.DateField()
    gender = sform.ChoiceField(choices=GENDER_CHOICES)
    country = sform.CharField(max_length=255)
    city = sform.CharField(max_length=255)

    @property
    def _is_merge(self):
        return self.initial.get('_mode') == CANDIDATE_UPLOAD_MODES.merge

    def get_field_state(self, name):
        # Поле original обязательное, если мы в режиме мёржа
        if name == 'original' and self._is_merge:
            return sform.REQUIRED
        # Поля ниже обязательны всегда, кроме режима мёржа
        if name in ('first_name', 'last_name') and not self._is_merge:
            return sform.REQUIRED
        return super().get_field_state(name)


class CandidateBulkUploadSheetForm(CandidateBulkUploadBaseForm):
    """
    Форма для массовой загрузки кандидатов через excel/csv/YT
    """
    target_cities = CommaSeparatedMappingField('cities')

    email = sform.EmailField()
    phone = sform.CharField(
        max_length=50,
        validators=[validate_phone],
    )
    vacancies_mailing_agreement = NullYesField()

    # Для обратной совместимости здесь contacts
    # – это список контактов с типом "other" через запятую,
    # а contact_list – это уже нормальный список словарей с возможностью задать тип
    contacts = CommaSeparatedField()
    contact_list = sform.GridField(sform.FieldsetField(CandidateContactForm))

    professions = CommaSeparatedMappingField('professions')
    skills = CommaSeparatedMappingField('skills', case_sensitive=False)
    tags = CommaSeparatedField()
    attachment_link = CommaSeparatedField()

    educations = sform.GridField(sform.FieldsetField(CandidateEducationForm))
    institution = sform.CharField(max_length=255)
    degree = sform.ChoiceField(CANDIDATE_DEGREES)
    end_date = PartialDateField()

    jobs = sform.GridField(sform.FieldsetField(CandidateJobForm))
    employer = sform.CharField(max_length=255)

    answers = sform.CharField()
    note = sform.CharField()
    note_author = UserSuggestField()

    # Системные поля, которые можно использовать только тем,
    # кому мы явно рассказали о них
    ah_modified_at = sform.DateTimeField()  # дата последнего синка с AH
    keep_modification_time = sform.BooleanField()  # не обновлять modified на кандидате

    def get_field_state(self, name):
        # Поле note_author обязательное, если есть заметка или ответы на вопросы
        if name == 'note_author':
            if self.cleaned_data.get('note') or self.cleaned_data.get('answers'):
                return sform.REQUIRED
        # Сохранение текущего modified на кандидате актуально только в режиме мёржа
        if name == 'keep_modification_time' and not self._is_merge:
            return sform.READONLY
        return super().get_field_state(name)

    def clean_answers(self, value):
        prefix = 'q_'
        answers = []
        for k in self.data:
            question = k.strip()
            answer = self.data[k].strip() if isinstance(self.data[k], str) else ''
            if question.startswith(prefix) and answer:
                answers.append({
                    'question': question.lstrip(prefix),
                    'answer': answer,
                })

        return answers

    def clean_original(self, original):
        if not original or not original.is_duplicate:
            return original
        else:
            return original.original

    def clean_attachment_link(self, urls):
        cleaned_urls = []
        url_validator = URLValidator()
        for url in urls:
            try:
                url_validator(url)
            except ValidationError:
                logger.error('Url %s is invalid', url)
                continue
            cleaned_urls.append(url)
        return cleaned_urls

    def clean(self):
        data = super().clean()

        # Перестраиваем контакты
        data['contacts'] = [_reformat_contact(c) for c in data.pop('contacts', [])]
        data['contacts'].extend(data.pop('contact_list', []))
        main_contacts = (
            (data.pop('email', None), CONTACT_TYPES.email),
            (data.pop('phone', None), CONTACT_TYPES.phone),
        )
        data['contacts'].extend([
            _reformat_contact(account_id, _type, is_main=True)
            for account_id, _type in main_contacts if account_id
        ])
        if not data['contacts'] and not self._is_merge:
            raise ValidationError({('contact_list',): Required()})

        # Перестраиваем образование
        data.setdefault('educations', [])
        institution = data.pop('institution', None)
        degree = data.pop('degree', None) or CANDIDATE_DEGREES.unknown
        end_date = data.pop('end_date', None)
        if institution:
            data['educations'].append({
                'institution': institution,
                'degree': degree,
                'end_date': end_date,
            })

        # Перестраиваем опыт работы
        data.setdefault('jobs', [])
        employer = data.pop('employer', None)
        if employer:
            data['jobs'].append({'employer': employer})

        # Перестраиваем профессии
        professions = data.pop('professions', [])
        data['candidate_professions'] = [
            {
                'profession': p,
                'professional_sphere_id': p.professional_sphere_id,
                'salary_expectation': '',
            }
            for p in professions
        ]

        return data


class CandidateBulkUploadStaffForm(CandidateBulkUploadBaseForm):
    """
    Форма для загрузки кандидатов со Стаффа
    """
    login = sform.CharField(max_length=64, state=sform.REQUIRED)
    contacts = ContactsField(state=sform.REQUIRED)
    note = sform.CharField()

    def clean(self):
        cleaned_data = super().clean()

        # По умолчанию заметки пишет robot-femida@
        users = self.initial.get('_mapping', {}).get('users', {})
        cleaned_data['note_author'] = users.get('robot-femida')

        for contact in cleaned_data.get('contacts', []):
            contact['source'] = CONTACT_SOURCES.staff

        return cleaned_data


class CandidateBulkUploadBeameryForm(CandidateBulkUploadBaseForm):
    """
    Форма для загрузки кандидатов из Beamery
    """
    beamery_id = sform.UUIDField(state=sform.REQUIRED)
    status = sform.ChoiceField(choices=CANDIDATE_STATUSES)
    contacts = ContactsField(state=sform.REQUIRED)
    educations = EducationsField()
    jobs = JobsField()
    tags = TagsField()

    def get_field_state(self, name):
        if name == 'first_name':
            return sform.NORMAL
        return super().get_field_state(name)

    def clean_original(self, original):
        if original and original.is_duplicate:
            raise SimpleValidationError('candidate_is_merged')
        return original

    def is_valid(self):
        if super().is_valid():
            return True
        # Считаем, что форма валидна, если валидны обязательные поля.
        # Невалидные необязательные поля игнорируем, но логгируем
        is_valid = all(
            field_name in self.cleaned_data
            for field_name in self.fields
            if self.get_field_state(field_name) == sform.REQUIRED
        )
        log_task = log_invalid_data_task if is_valid else log_fatally_invalid_data_task
        log_task.delay(self.data, self.errors_as_dict())
        return is_valid
