import waffle

from django.conf import settings
from django.template import loader
from rest_framework import serializers

from intranet.femida.src.candidates.choices import CONTACT_TYPES, CANDIDATE_DEGREES
from intranet.femida.src.core.serializers import FakeField
from intranet.femida.src.staff.contacts import fetch_contacts_from_person
from intranet.femida.src.staff.helpers import safe_getitem


BEAMERY_URL_TEMPLATE = 'https://app.beamery.ru/#/crm/profile/{id}'


class BeameryDateField(serializers.ReadOnlyField):

    def to_representation(self, value):
        # Beamery даже для дат отдаёт datetime, поэтому обрезаем всё после даты
        return value and value[:10]


class StaffPersonCandidateSerializer(serializers.Serializer):

    login = serializers.ReadOnlyField()
    first_name = serializers.ReadOnlyField(source='name.first.ru')
    middle_name = serializers.ReadOnlyField(source='name.middle.ru')
    last_name = serializers.ReadOnlyField(source='name.last.ru')
    gender = serializers.ReadOnlyField(source='personal.gender')
    birthday = serializers.ReadOnlyField(source='personal.birthday')
    contacts = serializers.SerializerMethodField()
    note = serializers.SerializerMethodField()

    def get_contacts(self, obj):
        return list(fetch_contacts_from_person(obj))

    def get_note(self, obj):
        org_id = safe_getitem(obj, ['official', 'organization', 'id'])
        if org_id in (settings.EXTERNAL_ORGANIZATION_ID, settings.YANDEX_MONEY_ORGANIZATION_ID):
            return loader.render_to_string(
                template_name='femida/candidates/ext-employee-sync-note.wiki',
                context={'person': obj},
            )


class BeameryCandidateEducationSerializer(serializers.Serializer):

    # TODO: уточнить, правильные ли тут маппинги; учесть обязательность полей
    institution = serializers.ReadOnlyField(source='organisationName')
    faculty = serializers.ReadOnlyField(source='program')
    # TODO: сделать маппинг по degree;
    degree = FakeField(CANDIDATE_DEGREES.unknown)
    end_date = BeameryDateField(source='endDate')


class BeameryCandidateJobSerializer(serializers.Serializer):

    employer = serializers.ReadOnlyField(source='organisationName')
    position = serializers.ReadOnlyField(source='role')
    start_date = BeameryDateField(source='startDate')
    end_date = BeameryDateField(source='endDate')


class BeameryCandidateSerializer(serializers.Serializer):

    original = serializers.ReadOnlyField(source='integrations.brassring.id', default=None)
    beamery_id = serializers.ReadOnlyField(source='id')
    status = serializers.ReadOnlyField(source='status.id', default=None)
    first_name = serializers.ReadOnlyField(source='firstName')
    middle_name = serializers.SerializerMethodField()
    last_name = serializers.ReadOnlyField(source='lastName')
    birthday = BeameryDateField(source='custom_fields.birthdate', default=None)
    city = serializers.ReadOnlyField(source='location.city', default=None)
    country = serializers.ReadOnlyField(source='location.country', default=None)

    tags = serializers.ReadOnlyField()
    contacts = serializers.SerializerMethodField()
    educations = BeameryCandidateEducationSerializer(source='education', many=True, read_only=True)
    jobs = BeameryCandidateJobSerializer(source='experience', many=True, read_only=True)

    # TODO: сделать маппинги ещё для пачки полей
    # ...

    def get_middle_name(self, obj):
        parts = obj.get('middleNames', [])
        return ' '.join(parts).strip() or None

    def get_contacts(self, obj):
        emails = obj.get('emails', [])
        phones = obj.get('phoneNumbers', [])
        contacts = [{
            'type': CONTACT_TYPES.beamery,
            'account_id': BEAMERY_URL_TEMPLATE.format(id=obj.get('id')),
        }]
        # TODO: перепроверить is_main
        contacts += [{'type': CONTACT_TYPES.email, 'account_id': i} for i in emails]
        contacts += [{'type': CONTACT_TYPES.phone, 'account_id': i} for i in phones]
        return contacts

    def to_representation(self, obj):
        self._reformat_custom_fields(obj)
        result = super().to_representation(obj)

        # FIXME: временно отключил, т.к. эти данные будут дублироваться
        #  через существующий механизм заливки
        if not waffle.switch_is_active('enable_upload_educations_from_beamery'):
            result.pop('educations', None)
        if not waffle.switch_is_active('enable_upload_jobs_from_beamery'):
            result.pop('jobs', None)

        return result

    def _reformat_custom_fields(self, obj):
        # Note: в customFields может прилететь список словарей в 1м из 2х форматов:
        # {'Name': 'Field Name', 'value': 'value'}
        # {'Name': 'List Field Name', 'values': ['1', '2']}
        obj['custom_fields'] = {}
        for field_data in obj.get('customFields', []):
            field = field_data['Name'].strip().lower().replace(' ', '_')
            obj['custom_fields'][field] = field_data.get('value') or field_data.get('values')
