import json
import os
import waffle
import xlwt
import yenv
import yt.wrapper as yt

from typing import Iterable, Dict

from django.conf import settings
from django.db import transaction
from django.utils.functional import cached_property

from intranet.femida.src.actionlog.models import actionlog
from intranet.femida.src.candidates.bulk_upload.choices import (
    CANDIDATE_UPLOAD_MODES,
    CANDIDATE_UPLOAD_RESOLUTIONS,
)
from intranet.femida.src.candidates.deduplication import DEFINITELY_DUPLICATE, MAYBE_DUPLICATE
from intranet.femida.src.candidates.deduplication.strategies import new_strategy, staff_strategy
from intranet.femida.src.candidates.bulk_upload.controllers import (
    create_candidate,
    merge_into_candidate,
    beamery_create_candidate,
    beamery_merge_into_candidate,
)
from intranet.femida.src.candidates.bulk_upload.forms import (
    CandidateBulkUploadBaseForm,
    CandidateBulkUploadSheetForm,
    CandidateBulkUploadStaffForm,
    CandidateBulkUploadBeameryForm,
)
from intranet.femida.src.candidates.bulk_upload.cache import CandidateDictionariesCache
from intranet.femida.src.candidates.bulk_upload.exceptions import InvalidFileExtension
from intranet.femida.src.candidates.bulk_upload.serializers import (
    StaffPersonCandidateSerializer,
)
from intranet.femida.src.candidates.deduplication.exceptions import DuplicateLoginConflictError
from intranet.femida.src.candidates.deduplication.shortcuts import get_most_likely_duplicate
from intranet.femida.src.users.models import get_robot_femida
from intranet.femida.src.utils.files import (
    SheetWriterDummyAdapter,
    SheetReaderCSVAdapter,
    SheetReaderXLSAdapter,
    SheetReaderYTAdapter,
    SheetWriterCSVAdapter,
    SheetWriterXLSAdapter,
    SheetWriterYTAdapter,
)
from intranet.femida.src.utils.urls import absolute_reverse


class CandidateBaseUploader:

    EMPTY_VALUE = None
    validator_class = CandidateBulkUploadBaseForm
    strategy = staticmethod(new_strategy)
    create_candidate = staticmethod(create_candidate)
    merge_into_candidate = staticmethod(merge_into_candidate)
    is_possible_duplicates_creation_enabled = True

    def __init__(self, mode, default_data=None, source=None):
        assert source, 'source cannot be empty'
        self.mode = mode
        self.default_data = default_data or {}
        self.source = source
        self.dry_run = mode == CANDIDATE_UPLOAD_MODES.check

    @property
    def reader(self) -> Iterable[Dict]:
        raise NotImplementedError

    @property
    def writer(self):
        raise NotImplementedError

    @cached_property
    def dictionaries(self):
        d = CandidateDictionariesCache()
        return {
            'cities': d.cities_by_name_map,
            'professions': d.professions_by_name_map,
            'skills': d.skills_by_name_map,
        }

    @transaction.atomic
    def upload(self, print_progress=False):
        writer = self.writer
        reader = self.reader

        for i, data in enumerate(reader, 1):
            result = self._process_raw_data(data)
            candidate = result['candidate']
            conflict_candidate = result['conflict_candidate']
            data['result'] = result['resolution']
            data['errors'] = self._format_json(result['errors'])
            data['candidate_id'] = self._format_int(candidate.id) if candidate else self.EMPTY_VALUE
            data['candidate_url'] = self._get_candidate_url(candidate)
            data['conflict_id'] = (
                self._format_int(conflict_candidate.id)
                if conflict_candidate
                else self.EMPTY_VALUE
            )
            data['conflict_url'] = self._get_candidate_url(conflict_candidate)
            writer.writerow(data)

            if print_progress and i % 100 == 0:
                print(f'Processed {i} rows')

        writer.close()

    def _format_int(self, data):
        return data

    def _format_json(self, data):
        return data

    def _process_raw_data(self, raw_data):
        """
        Принимает сырой словарь
        из одной строки табличного файла вида
        {
          "Заголовок": "Значение",
        }
        Возвращает словарь вида
        {
          "resolution": "resolution",
          "candidate": <итоговый кандидат>,
          "conflict_candidate": <потенциальный дубликат>,
          "errors": <sform.errors_dict>,
        }
        """
        # Проставляем значения по умолчанию
        # для отсутствующих и пустых колонок
        for k, v in self.default_data.items():
            if not raw_data.get(k, '').strip():
                raw_data[k] = v

        validator = self.validator_class(
            data=raw_data,
            initial={
                '_mode': self.mode,
                '_mapping': self.dictionaries,
            },
        )

        result = {
            'candidate': None,
            'conflict_candidate': None,
            'resolution': None,
            'errors': {},
        }

        if not validator.is_valid():
            result['resolution'] = CANDIDATE_UPLOAD_RESOLUTIONS.error
            result['errors'] = validator.errors_as_dict()
            return result

        try:
            result.update(self._update_or_create_candidate(validator.cleaned_data))
        except DuplicateLoginConflictError as exc:
            result['resolution'] = CANDIDATE_UPLOAD_RESOLUTIONS.error
            result['conflict_candidate'] = exc.duplicate
            result['errors'] = {'errors': {
                'login': [{'code': 'conflict'}],
            }}

        return result

    def _update_or_create_candidate(self, data):
        """
        Принимает решение, что нужно сделать на основе переданных данных:
        создать кандидата или добавить эти данные в существующего.
        После принятия решения действие выполняется.
        :return: {
            "resolution": "resolution",
            "candidate": <итоговый кандидат>,
            # может отсутствовать в результате
            "conflict_candidate": <потенциальный дубль>,
        }
        """
        result = {}
        duplicate, decision = self._get_duplicate_with_decision(data)
        if duplicate is None:
            result['candidate'] = self._create_candidate(data)
            result['resolution'] = CANDIDATE_UPLOAD_RESOLUTIONS.created
        elif decision == DEFINITELY_DUPLICATE:
            result['candidate'] = self._merge_into_candidate(duplicate, data)
            result['resolution'] = CANDIDATE_UPLOAD_RESOLUTIONS.merged
        else:
            if self.is_possible_duplicates_creation_enabled:
                result['candidate'] = self._create_candidate(data)
            result['resolution'] = CANDIDATE_UPLOAD_RESOLUTIONS.created_duplicate
            result['conflict_candidate'] = duplicate
        return result

    def _get_duplicate_with_decision(self, data):
        """
        Ищет дубликат, если необходимо, и отдаёт кортеж (duplicate, decision)
        Если загрузчик используется в режиме создания или мёржа кандидатов,
        дубли искать не пытаемся.
        """
        original = data.pop('original')
        if self.mode == CANDIDATE_UPLOAD_MODES.create:
            return None, None
        elif self.mode == CANDIDATE_UPLOAD_MODES.merge:
            return original, DEFINITELY_DUPLICATE

        duplicate, _, decision = get_most_likely_duplicate(
            candidate=data,
            strategy=self.strategy,
            threshold=MAYBE_DUPLICATE,
        )
        return duplicate, decision

    def _create_candidate(self, data):
        if not self.dry_run:
            with actionlog.init(f'{self.source}_candidate_create'):
                return self.create_candidate(data)

    def _merge_into_candidate(self, candidate, data):
        if self.dry_run:
            return candidate
        else:
            with actionlog.init(f'{self.source}_candidate_merge'):
                return self.merge_into_candidate(candidate, data)

    def _get_candidate_url(self, candidate):
        if not candidate:
            return self.EMPTY_VALUE
        return absolute_reverse(
            viewname='candidate-detail',
            kwargs={
                'candidate_id': candidate.id,
            },
        )


class CandidateSheetUploader(CandidateBaseUploader):

    EMPTY_VALUE = ''
    validator_class = CandidateBulkUploadSheetForm

    _file_readers = {
        'xls': SheetReaderXLSAdapter,
        'xlsx': SheetReaderXLSAdapter,
        'csv': SheetReaderCSVAdapter,
    }

    # Note: xlwt не умеет xlsx, поэтому пока
    # не даём выбирать его для output-файла
    _file_writers = {
        'xls': SheetWriterXLSAdapter,
        'csv': SheetWriterCSVAdapter,
    }

    _valid_input_extensions = set(_file_readers.keys())
    _valid_output_extensions = set(_file_writers.keys())

    def __init__(self, input_file_name, output_file_name='result.xls',
                 mode=CANDIDATE_UPLOAD_MODES.auto, default_data=None):
        self.input_file_name = self._validate_file_name(
            file_name=input_file_name,
            valid_extensions=self._valid_input_extensions,
        )
        self.output_file_name = self._validate_file_name(
            file_name=output_file_name,
            valid_extensions=self._valid_output_extensions,
        )
        super().__init__(mode, default_data, 'sheet')

    def _validate_file_name(self, file_name, valid_extensions):
        ext = os.path.splitext(file_name)[1][1:]
        if ext not in valid_extensions:
            raise InvalidFileExtension(file_name, valid_extensions)
        return file_name

    def _get_candidate_url(self, candidate):
        url = super()._get_candidate_url(candidate)
        if not url or self._is_csv_output:
            return url
        return xlwt.Formula('HYPERLINK("{url}";"{url}")'.format(url=url))

    @property
    def _is_csv_output(self):
        return self.output_file_name.endswith('.csv')

    @property
    def reader(self):
        ext = os.path.splitext(self.input_file_name)[1][1:]
        reader_class = self._file_readers[ext]
        return reader_class(self.input_file_name)

    @property
    def writer(self):
        ext = os.path.splitext(self.output_file_name)[1][1:]
        writer_class = self._file_writers[ext]
        return writer_class(self.output_file_name)

    def _format_int(self, data):
        return str(data)

    def _format_json(self, data):
        return json.dumps(data) if data else self.EMPTY_VALUE


class CandidateYTUploader(CandidateBaseUploader):

    validator_class = CandidateBulkUploadSheetForm

    def __init__(self, input_table_name, output_table_name=None,
                 mode=CANDIDATE_UPLOAD_MODES.check, source='yt'):
        self.input_table_name = input_table_name
        self.output_table_name = output_table_name or f'{input_table_name}-result'
        yt.config['token'] = settings.YT_TOKEN
        yt.config['proxy']['url'] = settings.YT_PROXY
        source = f'yt_{source}' if source else 'yt'
        super().__init__(mode, source=source)

    @property
    def reader(self):
        return SheetReaderYTAdapter(self.input_table_name)

    @property
    def writer(self):
        return SheetWriterYTAdapter(self.output_table_name, chunk_size=1000)

    def upload(self, print_progress=False):
        try:
            super().upload(print_progress)
        except Exception:
            if yt.exists(self.output_table_name):
                yt.remove(self.output_table_name)
            raise


class CandidateStaffUploader(CandidateBaseUploader):

    validator_class = CandidateBulkUploadStaffForm
    strategy = staticmethod(staff_strategy)

    def __init__(self, persons):
        self._persons = persons
        is_auto = waffle.switch_is_active('enable_staff_candidates_auto_sync')
        mode = CANDIDATE_UPLOAD_MODES.auto if is_auto else CANDIDATE_UPLOAD_MODES.check
        super().__init__(mode, source='staff')

    @property
    def is_possible_duplicates_creation_enabled(self):
        return waffle.switch_is_active('enable_staff_candidates_possible_duplicates_creation')

    @property
    def reader(self):
        for person in self._persons:
            data = StaffPersonCandidateSerializer(person).data
            # Загружаем в базу только людей с контактами
            if data['contacts']:
                yield data

    @cached_property
    def writer(self):
        # Note: записываем всё, что делаем, в YT-таблицу.
        # Как минимум, на первом этапе это поможет
        # принять решение по спорным кейсам
        yt.config['token'] = settings.YT_TOKEN
        yt.config['proxy']['url'] = settings.YT_PROXY
        env = 'test/' if yenv.type != 'production' else ''
        return SheetWriterYTAdapter(f'//home/femida/{env}staff-candidates', chunk_size=500)

    @cached_property
    def dictionaries(self):
        robot = get_robot_femida()
        return {
            'users': {
                robot.username: robot,
            },
        }

    def _merge_into_candidate(self, candidate, data):
        # Не пишем никаких заметок к существующим кандидатам
        data.pop('note', None)
        return super()._merge_into_candidate(candidate, data)


class CandidateBeameryUploader(CandidateBaseUploader):

    validator_class = CandidateBulkUploadBeameryForm
    create_candidate = staticmethod(beamery_create_candidate)
    merge_into_candidate = staticmethod(beamery_merge_into_candidate)

    def __init__(self, mode, rows: list[dict]):
        super().__init__(mode, source='beamery')
        self.rows = rows
        self.dry_run = not waffle.switch_is_active('enable_candidate_beamery_uploader')

    @property
    def reader(self):
        return iter(self.rows)

    @property
    def writer(self):
        return SheetWriterDummyAdapter()
