from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction

from intranet.femida.src.candidates.choices import (
    REFERENCE_EVENTS,
    CANDIDATE_STATUSES,
)
from intranet.femida.src.candidates.models import Candidate, Consideration
from intranet.femida.src.candidates.controllers import calculate_consideration_extended_status
from intranet.femida.src.candidates.signals import consideration_status_changed
from intranet.femida.src.candidates.tasks import send_candidate_reference_event_task
from intranet.femida.src.staff.models import Department


User = get_user_model()


def is_candidate_working_in_yandex(candidate_id):
    """
    Проверяет, работает ли кандидат в данный момент в подразделении Яндекс
    """
    return (
        User.objects
        .filter(
            department__in=Department.objects.in_tree(settings.YANDEX_DEPARTMENT_ID),
            username__in=Candidate.unsafe.alive().filter(id=candidate_id).values('login'),
            is_dismissed=False,
        )
        .exists()
    )


def get_relevant_consideration(candidate_id):
    """
    Получить актуальное рассмотрение для кандидата
    """
    return (
        Consideration.unsafe
        .filter(candidate_id=candidate_id)
        .filter(state=Consideration.STATES.in_progress)
        .first()
    )


def _send_consideration_created_reference_event(candidate_id, initiator_id):
    """
    Шлем эвент о создании рассмотрения, только если кандидат находится в работе
    """
    candidate_status = (
        Candidate.unsafe
        .filter(id=candidate_id)
        .values_list('status', flat=True)
        .first()
    )
    if candidate_status == CANDIDATE_STATUSES.in_progress:
        send_candidate_reference_event_task.delay(
            candidate_id=candidate_id,
            event=REFERENCE_EVENTS.consideration_created,
            initiator_id=initiator_id,
        )


def create_consideration(candidate_id, **params):
    # Переоткрываем REFERENCE-тикет, только если кандидат останется открытым до конца транзакции.
    # Так избегаем модификаций тикете при обработке быстрого отказа.
    transaction.on_commit(
        lambda: _send_consideration_created_reference_event(
            candidate_id=candidate_id,
            initiator_id=params['created_by'].id,
        )
    )

    if 'is_rotation' not in params:
        params['is_rotation'] = is_candidate_working_in_yandex(candidate_id)
    return Consideration.objects.create(candidate_id=candidate_id, **params)


def get_or_create_relevant_consideration(candidate, initiator, candidate_data=None):
    """
    Если кандидат открыт, отдает его актуальное рассмотрение.
    Если кандидат закрыт, открывает кандидата, создает новое рассмотрение и отдает его.
    """
    from intranet.femida.src.candidates.workflow import CandidateWorkflow

    candidate_data = candidate_data or {}
    action = CandidateWorkflow(candidate, initiator).get_action('open')

    # Note: считаем, что если пользователь добрался до сюда,
    # значит права на неявное открытие кандидата у него есть.
    # Проверяем только что кандидат в правильном статусе.
    if action.is_status_correct():
        action.perform(**candidate_data)
        return action.extra_data['consideration']
    else:
        consideration = get_relevant_consideration(candidate.id)

        # Note: у открытого кандидата может не быть активного рассмотрения,
        # потому что раньше этого не требовалось. Мы не стали при переходе
        # на новый workflow создавать активные рассмотрения для всех открытых кандидатов,
        # по которым ничего не происходит.
        if consideration is None:
            consideration = create_consideration(
                candidate_id=candidate.id,
                created_by=initiator,
            )

        responsibles = candidate_data.get('responsibles')
        if responsibles:
            responsibles = [user for user in responsibles if user != candidate.main_recruiter]
            consideration.candidate.responsibles.add(*responsibles)
        return consideration


def update_consideration_extended_status(consideration):
    extended_status = calculate_consideration_extended_status(consideration.id)
    if extended_status == consideration.extended_status:
        return
    consideration.extended_status = extended_status
    consideration.save(update_fields=['extended_status', 'modified'])
    consideration_status_changed.send(sender=Consideration, consideration=consideration)
