import logging
import re
import regex
import phonenumbers
import waffle

from django.conf import settings

from intranet.femida.src.candidates.choices import CONTACT_TYPES
from intranet.femida.src.core.exceptions import SimpleValidationError


logger = logging.getLogger(__name__)

HH_URL_PATTERN = r'.*/resume/(?P<hh_id>[a-zA-Z0-9]+)/?'
HH_RGX = re.compile(HH_URL_PATTERN, re.IGNORECASE)

AH_ESCAPED_HOST = settings.AMAZING_HIRING_HOST.replace('.', '\\.')
AH_URL_PATTERN = fr'.*{AH_ESCAPED_HOST}/profiles/(?P<ah_id>\d+)/?'
AH_RGX = re.compile(AH_URL_PATTERN, re.IGNORECASE)

LI_URL_PATTERN = r'.*linkedin.com/in/(?P<linkedin_id>[\p{L}\p{N}-]+)/?'
LI_RGX = regex.compile(LI_URL_PATTERN, re.IGNORECASE)

CONTACTS_CONFIG = {
    CONTACT_TYPES.hh: {
        'rgx': HH_RGX,
        'url': f'https://{settings.HH_HOST}/resume/',
    },
    CONTACT_TYPES.ah: {
        'rgx': AH_RGX,
        'url': f'https://{settings.AMAZING_HIRING_HOST}/profiles/',
    },
    CONTACT_TYPES.linkedin: {
        'rgx': LI_RGX,
        'url': 'https://www.linkedin.com/in/',
    }
}


def normalize_phone(account_id, region='RU'):
    try:
        parsed = phonenumbers.parse(account_id, region)
    except phonenumbers.NumberParseException as exc:
        logger.warning('Error during parse phone contact %s: %s', account_id, repr(exc))
        return

    if not phonenumbers.is_valid_number(parsed):
        logger.warning('Phone number is not valid: %s', account_id)
        return

    if not phonenumbers.is_possible_number(parsed):
        logger.warning('Phone number is not possible: %s', account_id)
        return

    return phonenumbers.format_number(
        numobj=parsed,
        num_format=phonenumbers.PhoneNumberFormat.INTERNATIONAL,
    )


def normalize_contact(contact_type, account_id):
    if account_id is None:
        return
    if contact_type == CONTACT_TYPES.phone:
        return normalize_phone(account_id)
    if contact_type in (CONTACT_TYPES.hh, CONTACT_TYPES.ah, CONTACT_TYPES.linkedin):
        return normalize_contact_url(contact_type, account_id)
    return account_id.strip().lower()


def get_or_create_contact(**data):
    """
    На основе data понимает, нужно создать новый контакт или восстановить удаленный
    """
    from intranet.femida.src.candidates.models import CandidateContact

    assert 'candidate_id' in data
    assert 'type' in data
    assert 'account_id' in data

    _filter_part = {
        'candidate_id': data['candidate_id'],
        'type': data['type'],
    }
    normalized_account_id = normalize_contact(
        contact_type=data['type'],
        account_id=data['account_id'],
    )
    # Если удалось нормализовать account_id, то ищем в базе по normalized_account_id
    # Если не вышло, то хотя бы проверим по наличию такого же сырого account_id
    if normalized_account_id:
        _filter_part['normalized_account_id'] = normalized_account_id
    else:
        _filter_part['account_id'] = data['account_id']

    # Контакты могут дублироваться, поэтому get_or_create здесь не подходит
    contact = CandidateContact.objects.filter(**_filter_part).order_by('-is_active').first()
    if contact is None:
        contact = CandidateContact.objects.create(**data)
    if not contact.is_active:
        contact.is_active = True
        contact.save()
    return contact


def parse_contact_id(contact_type, contact):
    rgx = CONTACTS_CONFIG[contact_type]['rgx']
    match = rgx.match(contact)
    result = match.groupdict() if match else {}
    return result.get(f'{contact_type}_id')


def normalize_contact_url(contact_type, contact):
    contact_id = parse_contact_id(contact_type, contact)
    if not contact_id:
        return None
    contact_id = contact_id.lower()
    return f'{CONTACTS_CONFIG[contact_type]["url"]}{contact_id}'


def validate_contact_url(contact_type, contact):
    normalized = normalize_contact_url(contact_type, contact)
    if normalized or not waffle.switch_is_active(f'enable_{contact_type}_contact_validation'):
        return
    raise SimpleValidationError(code=f'invalid_{contact_type}_contact')
