import waffle

from collections import defaultdict
from functools import cached_property

from django.db.models import OuterRef, Count, Case, When, F, Value, CharField, Q
from django.utils import timezone
from typing import List

from intranet.femida.src.candidates.choices import CHALLENGE_STATUSES, RKN_CONTACT_TYPES
from intranet.femida.src.communications.models import Message
from intranet.femida.src.interviews.choices import INTERVIEW_TYPES
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.candidates import models, choices
from intranet.femida.src.candidates.contacts import normalize_contact
from intranet.femida.src.candidates.signals import candidate_modified
from intranet.femida.src.core.db import RowToDict, RowsToList, OrderExpression
from intranet.femida.src.core.controllers import (
    update_instance,
    update_list_of_instances,
)
from intranet.femida.src.candidates.tasks import (
    run_duplicates_search_for_candidate,
    update_reference_issues_with_candidate_recruiters,
    update_rotation_issues_with_candidate_recruiters,
)
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.users.models import User
from intranet.femida.src.utils.switches import is_candidate_main_recruiter_enabled


CONSIDERATION_RECRUITER_STAGE_CONDITIONS = {
    k: Q(extended_status__in=v)
    for k, v in choices.CONSIDERATION_RECRUITER_STAGE_EXTENDED_STATUSES.items()
}


# FIXME: временное быстрое решение для редактирования тегов, пока не решим FEMIDA-5623
def _clear_prefetch_related(candidate: models.Candidate, key: str):
    """
    Очищает кэш объектов, полученных через prefetch
    """
    prefetch_cache = getattr(candidate, '_prefetched_objects_cache', {})
    if key in prefetch_cache:
        del prefetch_cache[key]


def update_candidate_contacts(candidate, contacts):
    if contacts is not None:
        for item in contacts:
            item['candidate'] = candidate
            item['is_active'] = True
            item['normalized_account_id'] = normalize_contact(
                contact_type=item['type'],
                account_id=item['account_id'],
            ) or ''
        if waffle.switch_is_active('is_rkn'):
            queryset = candidate.contacts.filter(type__in=RKN_CONTACT_TYPES._db_values)
        else:
            queryset = candidate.contacts.all()
        update_list_of_instances(
            model=models.CandidateContact,
            queryset=queryset,
            data=contacts,
            identifier=('type', 'normalized_account_id'),
            delete_func=lambda queryset: queryset.update(is_active=False),
        )


def update_candidate_educations(candidate, educations):
    if educations is not None:
        for item in educations:
            item['candidate'] = candidate
        update_list_of_instances(
            model=models.CandidateEducation,
            queryset=candidate.educations.all(),
            data=educations,
        )


def update_candidate_jobs(candidate, jobs):
    if jobs is not None:
        for item in jobs:
            item['candidate'] = candidate
        update_list_of_instances(
            model=models.CandidateJob,
            queryset=candidate.jobs.all(),
            data=jobs,
        )


def update_candidate_professions(candidate, candidate_professions):
    if candidate_professions is not None:
        professions_data = [
            {
                'candidate_id': candidate.id,
                'profession_id': item['profession'].id,
                'professional_sphere_id': item['profession'].professional_sphere_id,
                'salary_expectation': item['salary_expectation'],
            }
            for item in candidate_professions
        ]
        update_list_of_instances(
            model=models.CandidateProfession,
            queryset=candidate.candidate_professions.all(),
            data=professions_data,
            identifier=('candidate_id', 'profession_id'),
        )


def add_nonexistent_candidate_professions(candidate, professions):
    """
    Добавляем новые профессии без удаления старых
    """
    if professions is not None:
        new_professions_data = (
            professions
            .exclude(id__in=candidate.candidate_professions.values('profession_id'))
            .values('id', 'professional_sphere_id')
        )
        models.CandidateProfession.objects.bulk_create([
            models.CandidateProfession(
                candidate_id=candidate.id,
                profession_id=item['id'],
                professional_sphere_id=item['professional_sphere_id'],
            )
            for item in new_professions_data
        ])


def update_candidate_target_cities(candidate, cities):
    if cities is not None:
        cities_data = [
            {
                'candidate_id': candidate.id,
                'city_id': city.id
            }
            for city in cities
        ]
        update_list_of_instances(
            model=models.CandidateCity,
            queryset=candidate.candidate_cities.all(),
            data=cities_data,
            identifier=('candidate_id', 'city_id'),
        )


def update_candidate_skills(candidate, skills):
    if skills is not None:
        skills_data = [
            {
                'candidate_id': candidate.id,
                'skill_id': skill.id,
                'confirmed_by': [],
            }
            for skill in skills
        ]
        update_list_of_instances(
            model=models.CandidateSkill,
            queryset=candidate.candidate_skills.all(),
            data=skills_data,
            identifier=('candidate_id', 'skill_id'),
        )


def update_candidate_tags(candidate, tags, delete_missing=True):
    if tags is not None:
        tags_data = [
            {
                'candidate_id': candidate.id,
                'tag_id': tag.id,
                'is_active': True,
            }
            for tag in tags
        ]
        update_list_of_instances(
            model=models.CandidateTag,
            queryset=candidate.candidate_tags.all(),
            data=tags_data,
            identifier=('candidate_id', 'tag_id'),
            delete_missing=delete_missing,
            delete_func=lambda queryset: queryset.update(is_active=False),
        )
        _clear_prefetch_related(candidate, 'candidate_tags')


def update_candidate_attachments(candidate, attachments):
    if attachments is not None:
        attachments_data = [
            {
                'attachment_id': a.id,
                'candidate_id': candidate.id,
                'type': choices.ATTACHMENT_TYPES.resume,
            }
            for a in attachments
        ]
        update_list_of_instances(
            model=models.CandidateAttachment,
            queryset=candidate.candidate_attachments.all(),
            data=attachments_data,
            identifier=('candidate_id', 'attachment_id'),
        )


def update_issues_with_new_candidate_recruiters(candidate, roles):
    if not roles:
        return

    data = {}
    if choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter in roles:
        data['main_recruiter'] = candidate.main_recruiter.username
    if choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter in roles:
        data['recruiters'] = [recruiter.username for recruiter in candidate.recruiters]

    update_reference_issues_with_candidate_recruiters.delay(candidate.id, **data)
    update_rotation_issues_with_candidate_recruiters.delay(candidate.id, **data)


def update_candidate_responsibles(candidate, responsibles_by_role):
    responsibles_data = []
    roles = set()
    main_recruiter = responsibles_by_role['main_recruiter']
    if main_recruiter is not None:
        roles.add(choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter)
        responsibles_data.append({
            'candidate_id': candidate.id,
            'user_id': responsibles_by_role['main_recruiter'].id,
            'role': choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter,
        })

    recruiters = responsibles_by_role['recruiters']
    if recruiters is not None:
        roles.add(choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter)
        for recruiter in set(recruiters) - {main_recruiter}:
            responsibles_data.append({
                'candidate_id': candidate.id,
                'user_id': recruiter.id,
                'role': choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter,
            })
    update_list_of_instances(
        model=models.CandidateResponsible,
        queryset=candidate.candidate_responsibles.filter(role__in=roles),
        data=responsibles_data,
        identifier=('candidate_id', 'user_id', 'role'),
    )

    has_cached_responsibles = 'responsibles_by_role' in candidate.__dict__
    if has_cached_responsibles and roles:
        del candidate.responsibles_by_role
        _clear_prefetch_related(candidate, 'candidate_responsibles')

    update_issues_with_new_candidate_recruiters(candidate, roles)


def update_or_create_candidate(data, initiator=None, instance=None):
    contacts = data.pop('contacts', None)
    educations = data.pop('educations', None)
    jobs = data.pop('jobs', None)
    candidate_professions = data.pop('candidate_professions', None)
    new_professions = data.pop('new_professions', None)
    target_cities = data.pop('target_cities', None)
    skills = data.pop('skills', None)
    tags = data.pop('tags', None)
    main_recruiter = data.pop('main_recruiter', None)
    responsibles = data.pop('responsibles', None)
    recruiters = data.pop('recruiters', None)
    attachments = data.pop('attachments', None)

    data.pop('status', None)

    if instance:
        # Если поменялся контакт или образование, мы хотим, чтобы кандидат тоже пересохранялся,
        # чтобы был актуален modified
        instance = update_instance(instance, data, extra_update_fields=['modified'])
        candidate_modified.send(sender=models.Candidate, candidate=instance)
    else:
        responsibles = list(responsibles or [])
        data['created_by'] = initiator
        is_initiator_implicit_responsible = (
            initiator
            and not initiator.is_robot_femida
            and initiator not in responsibles
            and not is_candidate_main_recruiter_enabled()
        )
        if is_initiator_implicit_responsible:
            responsibles.append(initiator)

        if contacts and any(c['type'] == choices.CONTACT_TYPES.ah for c in contacts):
            data['ah_modified_at'] = timezone.now()

        instance = models.Candidate.objects.create(**data)
        if initiator and is_candidate_main_recruiter_enabled() and not main_recruiter:
            main_recruiter = initiator

    update_candidate_contacts(instance, contacts)
    update_candidate_educations(instance, educations)
    update_candidate_jobs(instance, jobs)
    update_candidate_professions(instance, candidate_professions)
    add_nonexistent_candidate_professions(instance, new_professions)
    update_candidate_target_cities(instance, target_cities)
    update_candidate_skills(instance, skills)
    update_candidate_tags(instance, tags)
    update_candidate_attachments(instance, attachments)

    responsibles_by_role = {
        'main_recruiter': main_recruiter,
        'recruiters': recruiters if recruiters is not None else responsibles
    }
    update_candidate_responsibles(instance, responsibles_by_role)

    run_duplicates_search_for_candidate.delay(instance.id)

    return instance


def get_active_consideration_subquery():
    return get_last_consideration_subquery(only_active=True)


def get_last_consideration_subquery(only_active=False):
    filter_data = {
        'candidate_id': OuterRef('id'),
        'is_last': True,
    }
    if only_active:
        filter_data['state'] = choices.CONSIDERATION_STATUSES.in_progress
    return RowToDict(
        models.Consideration.unsafe
        .filter(**filter_data)
        .values(
            'id',
            'state',
            'extended_status',
            'started',
            'modified',
            'finished',
        )[:1]
    )


def get_candidates_extended_statuses(candidate_ids):
    result = {}
    qs = models.Candidate.unsafe.filter(id__in=candidate_ids)
    qs = (
        qs.annotate(active_consideration=get_active_consideration_subquery())
        .values_list('id', 'status', 'active_consideration')
    )
    for candidate_id, status, consideration in qs:
        if consideration is not None:
            result[candidate_id] = consideration.get('extended_status')
        elif status == choices.CANDIDATE_STATUSES.in_progress:
            result[candidate_id] = choices.CONSIDERATION_EXTENDED_STATUSES.in_progress
        else:
            result[candidate_id] = choices.CONSIDERATION_EXTENDED_STATUSES.archived
    return result


def get_candidates_extended_statuses_and_changed_at(candidate_ids):
    result = defaultdict(dict)
    qs = models.Candidate.unsafe.filter(id__in=candidate_ids)
    qs = (
        qs.annotate(last_consideration=get_last_consideration_subquery())
        .values_list('id', 'status', 'last_consideration')
    )
    for candidate_id, status, consideration in qs:
        is_consideration_in_progress = (
            consideration is not None
            and consideration.get('state') == choices.CONSIDERATION_STATUSES.in_progress
        )
        if is_consideration_in_progress:
            extended_status = consideration.get('extended_status')
        elif status == choices.CANDIDATE_STATUSES.in_progress:
            extended_status = choices.CONSIDERATION_EXTENDED_STATUSES.in_progress
        else:
            extended_status = choices.CONSIDERATION_EXTENDED_STATUSES.archived
        changed_at = consideration.get('modified') if consideration else None
        result[candidate_id]['extended_status'] = extended_status
        result[candidate_id]['changed_at'] = changed_at
    return result


def get_candidate_message_counts(candidate_id):
    return dict(
        Message.unsafe
        .alive()
        .filter(candidate_id=candidate_id)
        .values_list('type')
        .annotate(count=Count('type'))
    )


def _calculate_extended_status_by_offers(offers):
    offer_statuses = {offer['status'] for offer in offers}
    ordered_statuses = reversed([i for i, _ in choices.CONSIDERATION_EXTENDED_STATUSES_WITH_OFFER])
    for status in ordered_statuses:
        if choices.OFFER_STATUSES_BY_CONSIDERATION_STATUS[status] & offer_statuses:
            return status


def _calculate_extended_status_by_interviews(interviews):
    REGULAR_OR_AA = 'interview'

    ordered_interview_types = (
        INTERVIEW_TYPES.final,
        REGULAR_OR_AA,
        INTERVIEW_TYPES.screening,
        INTERVIEW_TYPES.hr_screening,
    )
    interview_statuses_by_type = defaultdict(set)
    for i in interviews:
        if i['type'] in (INTERVIEW_TYPES.regular, INTERVIEW_TYPES.aa):
            _type = REGULAR_OR_AA
        else:
            _type = i['type']
        interview_statuses_by_type[_type].add(i['state'])

    for _type in ordered_interview_types:
        statuses = interview_statuses_by_type[_type]
        if {Interview.STATES.assigned, Interview.STATES.estimated} & statuses:
            return _type + '_assigned'
        if Interview.STATES.finished in statuses:
            return _type + '_finished'


def _calculate_extended_status_by_challenges(challenges):
    challenge_statuses = {ch['status'] for ch in challenges}
    ordered_challenge_statuses = (
        CHALLENGE_STATUSES.pending_review,
        CHALLENGE_STATUSES.assigned,
        CHALLENGE_STATUSES.finished,
    )
    for status in ordered_challenge_statuses:
        if status in challenge_statuses:
            return 'challenge_' + status


def _get_consideration_qs(ids):
    return (
        models.Consideration.unsafe
        .filter(id__in=ids)
        .annotate(
            offers_attr=RowsToList(
                Offer.unsafe.alive()
                .filter(candidate_id=OuterRef('candidate_id'))
                .values('status')
            ),
            interviews_attr=RowsToList(
                Interview.unsafe.alive()
                .filter(consideration_id=OuterRef('id'))
                .values('type', 'state')
            ),
            challenges_attr=RowsToList(
                models.Challenge.objects.alive()
                .filter(consideration_id=OuterRef('id'))
                .values('status')
            ),
        )
        .values('id', 'state', 'offers_attr', 'interviews_attr', 'challenges_attr')
    )


def bulk_calculate_consideration_extended_statuses(consideration_ids, with_offer=None):
    if with_offer is None:
        with_offer = waffle.switch_is_active('enable_offers_in_extended_statuses')
    result = {}
    for cons in _get_consideration_qs(consideration_ids):
        if cons['state'] == choices.CONSIDERATION_EXTENDED_STATUSES.archived:
            result[cons['id']] = choices.CONSIDERATION_EXTENDED_STATUSES.archived
        elif with_offer and cons['offers_attr']:
            result[cons['id']] = _calculate_extended_status_by_offers(cons['offers_attr'])
        elif cons['interviews_attr']:
            result[cons['id']] = _calculate_extended_status_by_interviews(cons['interviews_attr'])
        elif cons['challenges_attr']:
            result[cons['id']] = _calculate_extended_status_by_challenges(cons['challenges_attr'])
        else:
            result[cons['id']] = choices.CONSIDERATION_EXTENDED_STATUSES.in_progress
    return result


def calculate_consideration_extended_status(consideration_id):
    """
    Возвращает расширенный статус рассмотрения.
    https://wiki.yandex-team.ru/intranet/femida/interviews/#sostojanijarassmotrenijj
    """
    return bulk_calculate_consideration_extended_statuses([consideration_id]).get(consideration_id)


def add_extended_status_to_candidates_qs(qs):
    """
    Добавляет extended_status кандидатам на основе рассмотрения.
    Запрос упрощенный. С учетом того, что он всегда применяется только для
    кандидатов пофильтрованных по considerations__state='in_progress',
    и мы знаем, что у кандидата может быть только одно активное рассмотрение.

    Если понадобится добавлять реальный extended_status к любому кандидату,
    с учетом всех его рассмотрений, эта функция в таком виде не подойдет.
    """
    return qs.annotate(extended_status=F('considerations__extended_status'))


def add_extended_status_order_to_candidates_qs(qs):
    if 'extended_status' not in qs.query.annotations:
        qs = add_extended_status_to_candidates_qs(qs)

    return qs.annotate(
        extended_status_order=OrderExpression(
            field_name='extended_status',
            choices=choices.CONSIDERATION_EXTENDED_STATUSES,
        ),
    )


def add_recruiter_stage_to_candidates_qs(qs):
    if 'extended_status' not in qs.query.annotations:
        qs = add_extended_status_to_candidates_qs(qs)

    stage_cases = [
        When(condition, then=Value(stage))
        for stage, condition
        in CONSIDERATION_RECRUITER_STAGE_CONDITIONS.items()
    ]
    return qs.annotate(
        stage=Case(
            default=Value(''),
            output_field=CharField(),
            *stage_cases
        ),
    )


def count_candidates_per_recruiter_stage(qs):
    if 'stage' not in qs.query.annotations:
        qs = add_recruiter_stage_to_candidates_qs(qs)

    return dict(
        qs
        .values('stage')
        .annotate(count=Count('id'))
        .values_list('stage', 'count')
    )


def count_candidates_per_responsible_role(qs, user):
    return dict(
        models.CandidateResponsible.objects
        .filter(candidate__in=qs, user=user)
        .values('role')
        .annotate(count=Count('candidate_id'))
        .values_list('role', 'count')
    )


def filter_candidates_by_recruiter_stage(qs, stage):
    return qs.filter(CONSIDERATION_RECRUITER_STAGE_CONDITIONS[stage])


def filter_candidates_by_responsible_role(qs, user, role):
    return qs.filter(
        candidate_responsibles__user=user,
        candidate_responsibles__role=role,
    )


class CandidateCostSetNotFound(Exception):
    pass


class CandidateCostsSetMustContainsAtLeastOneCost(Exception):
    pass


class UserHasNoAccessToCandidateCostsSet(Exception):
    pass


class CandidateCostHistory:

    def __init__(self, candidate: models.Candidate):
        self._candidate = candidate

    def all(self):
        return self._costs_list

    def get_last(self):
        if self._costs_list:
            return self._costs_list[0]
        return None

    def get_history(self):
        return self._costs_list[1:]

    @cached_property
    def _costs_list(self) -> [models.CandidateCostsSet]:
        return list(
            self._candidate.candidate_costs_set
            .order_by('-created')
            .prefetch_related('costs', 'costs__currency')
            .select_related('created_by')
        )

    def edit_from_form(self, user: User, form_data: dict):
        new_costs = self._make_cost_list(form_data)
        if not new_costs:
            raise CandidateCostsSetMustContainsAtLeastOneCost
        current_cost_set = self.get_last()
        if current_cost_set and current_cost_set.created_by_id == user.id:
            self._edit(current_cost_set, new_costs)
        elif current_cost_set:
            raise UserHasNoAccessToCandidateCostsSet
        else:
            raise CandidateCostSetNotFound

    def update_from_form(self, user: User, form_data: dict) -> None:
        cost_list = self._make_cost_list(form_data)
        if not cost_list:
            raise CandidateCostsSetMustContainsAtLeastOneCost
        self._update_cost_set(cost_list, user)

    def _edit(self, costs_set: models.CandidateCostsSet, costs_list: List[models.CandidateCost]):
        # TODO: Возможно, тут в будущем стоит подумать о более щадящем редактираовнии
        costs_set.costs.all().delete()
        self._append_costs_to_cost_set(costs_set, costs_list)
        costs_set.save(update_fields=['modified'])

    def _update_cost_set(self, costs_list: List[models.CandidateCost], user: User):
        costs_set = models.CandidateCostsSet(candidate=self._candidate, created_by=user)
        costs_set.save()
        self._append_costs_to_cost_set(costs_set, costs_list)

    @staticmethod
    def _append_costs_to_cost_set(costs_set: models.CandidateCostsSet, costs_list: List[models.CandidateCost]):
        for cost in costs_list:
            cost.candidate_costs_set = costs_set
        models.CandidateCost.objects.bulk_create(costs_list)

    def _make_cost_list(self, form_data: dict) -> List[models.CandidateCost]:
        return [
            models.CandidateCost(cost_group=cost_group, type=_type, **values)
            for cost_group, costs in form_data.items()
            for _type, values in costs.items()
            if values
        ]

    def copy_from_candidates(self, candidates: List[models.Candidate]):
        costs_sets = list(
            models.CandidateCostsSet.objects.filter(candidate__in=candidates)
            .order_by('created')
            .prefetch_related('costs')
        )
        new_costs = []
        for costs_set in costs_sets:
            costs = list(costs_set.costs.all())
            costs_set.id = None
            costs_set.candidate = self._candidate
            costs_set.save()
            for cost in costs:
                cost.id = None
                cost.candidate_costs_set = costs_set
                new_costs.append(cost)
        models.CandidateCost.objects.bulk_create(new_costs)
