import logging
from functools import total_ordering

from django.utils.functional import cached_property

from .adapters import to_cand_adapter


logger = logging.getLogger(__name__)


_is_empty = lambda x: x is None or x == ''


@total_ordering
class SimilarityInfo:
    """
    Результат сравнения двух кандидатов. На выходе получаем:
        - score - примерный уровень похожести кандидатов, полезен для сортировки
            - За совпадение имени даем 0.5
            - За совпадение фамилии - 1.0
            - За совпадение контакта или хеша аттача - 2.0
        - score_details - распределение score по полям
        - match_details - сколько совпадений для каждого поля
            (для простых полей 0|1, для коллекций 0+)
        - conflict_fields - множество конфликтующих полей, не позволяющий очевидно их смержить
    """
    SIMPLE_FIELDS_WEIGHTS = {
        'first_name': 0.5,
        'middle_name': 0.0,
        'last_name': 1.0,
        'birthday': 0.0,
        'gender': 0.0,
        'country': 0.0,
        'city': 0.0,
        'source': 0.0,
        'source_description': 0.0,
        'login': 0.0,
        'inn': 0.0,
    }

    COLLECTION_FIELDS_WEIGHTS = {
        'contacts': 2.0,
        'attachment_hashes': 2.0,
    }

    def __init__(self, first, second, *args, **kwargs):
        self.first_candidate = first
        self.second_candidate = second
        self.first = to_cand_adapter(first)
        self.second = to_cand_adapter(second)

    @cached_property
    def _calculation(self):
        result = {
            'score': 0.0,
            'score_details': {},
            'match_details': {},
            'conflict_fields': set(),
        }

        for field_name, weight in self.SIMPLE_FIELDS_WEIGHTS.items():
            value1 = getattr(self.first, field_name)
            value2 = getattr(self.second, field_name)

            # Если хотя бы одно значение пустое (None или пустая строка),
            # то конфликтов или совпадений нет
            if _is_empty(value1) or _is_empty(value2):
                result['match_details'][field_name] = 0
                continue

            if value1 == value2:
                result['score'] += weight
                result['match_details'][field_name] = 1
                if weight:
                    result['score_details'][field_name] = weight
            else:
                result['conflict_fields'].add(field_name)
                result['match_details'][field_name] = 0

        for field_name, weight in self.COLLECTION_FIELDS_WEIGHTS.items():
            value1 = getattr(self.first, field_name)
            value2 = getattr(self.second, field_name)
            number_of_matches = len(value1 & value2)
            score = weight * number_of_matches
            result['score'] += score
            result['score_details'][field_name] = score
            result['match_details'][field_name] = number_of_matches

        return result

    @property
    def score(self):
        return self._calculation['score']

    @property
    def score_details(self):
        return self._calculation['score_details']

    @property
    def match_details(self):
        return self._calculation['match_details']

    @property
    def conflict_fields(self):
        return self._calculation['conflict_fields']

    def _get_contacts_str(self, contacts):
        return '\n'.join('    {c[0]}: {c[1]}'.format(c=c) for c in contacts)

    def _get_candidate_str(self, cand):
        s = (
            'ID: {cand.id}\nNAME: {cand.first_name} {cand.last_name}\n'
            'CONTACTS:\n{contacts_str}'
        )
        return s.format(
            cand=cand,
            contacts_str=self._get_contacts_str(cand.contacts),
        )

    def to_dict(self):
        return self._calculation

    def __str__(self):
        return '{divider}\n{first}\n\n{second}\n{divider}\n'.format(
            divider='-' * 40,
            first=self._get_candidate_str(self.first),
            second=self._get_candidate_str(self.second),
        )

    def __eq__(self, other):
        return self.score == other.score

    def __lt__(self, other):
        return self.score < other.score
