import json

import sform

from functools import reduce
from operator import and_

from constance import config

from intranet.femida.src.candidates.choices import CANDIDATE_SORTING_TYPES
from intranet.femida.src.candidates.filters import fields # noqa
from intranet.femida.src.candidates.filters.fields.base import filter_registry
from intranet.femida.src.core.choices import get_partial_choices


def get_choices():
    disabled_filters = json.loads(config.DISABLED_FILTERS)
    return [(f, f) for f in filter_registry if f not in disabled_filters]


class FilterForm(sform.SForm):

    field = sform.ChoiceField(
        choices=[(f, f) for f in filter_registry],
        state=sform.REQUIRED,
        default='SkillFilter',
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['field'].choices = get_choices()

    def data_as_dict(self, prefix=''):
        data = super().data_as_dict(prefix)

        filter_name = self._get_initial_value('field', self.fields['field'])
        field = filter_registry[filter_name]
        condition_form = field.get_form_cls()(initial=self.initial)
        data.update(condition_form.data_as_dict(prefix))

        condition_name = condition_form._get_initial_value(
            'condition',
            condition_form.fields['condition'],
        )
        condition = field.conditions[condition_name]
        query_form = condition.form_cls(initial=self.initial, context={'condition': condition})
        data.update(query_form.data_as_dict(prefix))

        return data

    def clean(self):
        filter_name = self.cleaned_data.get('field')
        if not filter_name:
            return self.cleaned_data

        field = filter_registry[filter_name]
        condition_form = field.get_form_cls()(self.data)
        errors = None

        if condition_form.is_valid():
            condition = field.conditions[condition_form.cleaned_data['condition']]
            query_form = condition.form_cls(self.data, context={'condition': condition})
            if query_form.is_valid():
                self.cleaned_data['query'] = query_form.cleaned_data
                self.cleaned_data['condition'] = condition(cleaned_data=query_form.cleaned_data)
            else:
                errors = query_form._get_errors()
        else:
            errors = condition_form._get_errors()

        if errors:
            self._errors.update(errors)

        return self.cleaned_data


class FiltersForm(sform.SForm):

    filters = sform.GridField(
        sform.FieldsetField(FilterForm),
        state=sform.REQUIRED,
    )
    sort = sform.ChoiceField(
        choices=get_partial_choices(
            CANDIDATE_SORTING_TYPES,
            CANDIDATE_SORTING_TYPES.modified_asc,
            CANDIDATE_SORTING_TYPES.modified_desc,
        ),
    )


class FilterCtl(object):
    """
    Принимает следующую структуру вида:
    {
        'filters':[
            {
                'field': 'ProfessionFilter',
                'condition': 'In',
                'values': [1],
            },
            {
                'field': 'SkillFilter',
                'condition': 'Exclude',
                'values': [2,3],
            },
        ],
        'sort': 'modified_asc',
    }

    где:
    - 'filters' это список словарей

        - field - имя фильтра, которое определяет класс фильтра
        - condition - класс условия для фильтра
        - остальные поля - параметры для условия

    - 'sort' - поле определяющее сортировку и ее направление.
    """

    def __init__(self, data=None, **kwargs):
        initial = kwargs.get('initial')
        if initial and not data:
            self.data = initial
        else:
            self.data = data
        self._form = None

    @classmethod
    def as_meta_dict(cls, initial=None):
        result = FiltersForm(initial=initial).as_dict()
        result['conditions'] = {
            name: field_filter.as_meta_dict()
            for name, field_filter in filter_registry.items()
        }
        return result

    def as_dict(self):
        return self.as_meta_dict(initial=self.data)

    def is_valid(self):
        if self._form is None:
            self._form = FiltersForm(self.data)
        return self._form.is_valid()

    @property
    def errors(self):
        return self._form.errors

    def get_query(self):
        queries = (f['condition'].q() for f in self._form.cleaned_data['filters'])
        return reduce(and_, queries)

    def get_annotations(self):
        annotations = (f['condition'].get_annotations() for f in self._form.cleaned_data['filters'])
        return reduce(lambda a, b: {**a, **b}, annotations)
