from copy import deepcopy

import sform

from intranet.femida.src.permissions.helpers import get_manager
from intranet.femida.src.permissions.managers.candidate import CandidatePermQuery

filter_registry = {}


def register_filter(klass):
    assert klass.__name__ not in filter_registry
    filter_registry[klass.__name__] = klass
    if hasattr(klass, 'generate_conditions'):
        klass.generate_conditions()
    return klass


class FieldFilter:

    conditions = None
    _form_cls = None
    model_class = None
    model_candidate_attr = 'candidate_id'
    form_base = sform.SForm
    form_fields = {}
    condition_classes = []

    @classmethod
    def register_condition(cls, con):
        if cls.conditions is None:
            cls.conditions = {}

        assert con.__name__ not in cls.conditions
        cls.conditions[con.__name__] = con
        con.filter = cls
        return con

    @classmethod
    def as_meta_dict(cls):
        result = cls.get_form_cls()().structure_as_dict(prefix='filters[0]')
        forms = {}
        for name, condition in cls.conditions.items():
            context = {'condition': condition}
            forms[name] = condition.form_cls(context=context).structure_as_dict(prefix='filters[0]')

        result['forms'] = forms
        return result

    @classmethod
    def get_form_cls(cls):
        if cls._form_cls is None:
            conditions = list(cls.conditions.keys())

            class ConditionForm(sform.SForm):
                condition = sform.ChoiceField(
                    choices=[(name, name) for name in conditions],
                    state=sform.REQUIRED,
                    default=conditions[0],
                )
            cls._form_cls = ConditionForm
        return cls._form_cls

    @classmethod
    def generate_conditions(cls):
        form = type('Form', (cls.form_base,), deepcopy(cls.form_fields))
        for klass in cls.condition_classes:
            cls.register_condition(klass.get_cls_with_form(form))

    @classmethod
    def get_queryset(cls):
        # Note: Доступы к кандидатам проверяются в ручке фильтров
        # Если доступы у сущности, по которой фильтруем, совпадают с доступами к кандидату
        # то второй раз вычислять доступы не нужно, и можно брать unsafe
        unsafe = (
            hasattr(cls.model_class.objects, '_perm_query_cls')
            and issubclass(cls.model_class.objects._perm_query_cls, CandidatePermQuery)
        )
        return get_manager(cls.model_class, unsafe=unsafe).all()


class Condition:

    filter = FieldFilter
    form_cls = None

    @classmethod
    def get_cls_with_form(cls, form_cls):
        if cls.form_cls is not None:
            return cls
        return type(cls.__name__, (cls,), {'form_cls': form_cls})

    def __init__(self, cleaned_data=None):
        self.cleaned_data = cleaned_data

    def get_annotations(self):
        return {}

    def q(self):
        raise NotImplementedError
