import sform

from django.db.models import Case, When, F, Subquery

from intranet.femida.src.api.core.forms import UserMultipleSuggestField
from intranet.femida.src.candidates.choices import (
    CONSIDERATION_STATUSES,
    CONSIDERATION_RESOLUTIONS,
    CONSIDERATION_EXTENDED_STATUSES,
)
from intranet.femida.src.candidates.filters.fields.base import register_filter
from intranet.femida.src.candidates.filters.fields.base.multiple import (
    MultipleChoiceFilter,
    Any,
    NotAny,
)
from intranet.femida.src.candidates.filters.fields.base.range import DateRangeFilter
from intranet.femida.src.candidates.helpers import get_extended_status_changed_at_subquery
from intranet.femida.src.candidates.models import (
    Consideration,
    CandidateResponsible,
    ConsiderationResponsible,
)
from intranet.femida.src.offers.choices import SOURCES


class ConsiderationAnnotateMixin:

    @classmethod
    def get_queryset(cls):
        assert cls.model_value_attr.startswith('_')
        attr = cls.model_value_attr[1:]
        return (
            Consideration.unsafe
            .annotate(**{
                cls.model_value_attr: Case(
                    When(state=CONSIDERATION_STATUSES.in_progress, then=F(f'candidate__{attr}')),
                    default=F(attr),
                ),
            })
        )


class ConsiderationSourceFilterBase(ConsiderationAnnotateMixin, MultipleChoiceFilter):

    model_class = Consideration
    model_value_attr = '_source'
    values_field = sform.MultipleChoiceField(state=sform.REQUIRED, choices=SOURCES)


class ConsiderationRecruiterFilterBase(MultipleChoiceFilter):

    model_value_attr = 'user'
    values_field = UserMultipleSuggestField(state=sform.REQUIRED)

    @classmethod
    def get_queryset(cls):
        """
        Возвращает объект с методом filter, в котором возвращается QuerySet c union данных
        CandidateResponsible и ConsiderationResponsible
        Если делать все джоины в одном подзапросе (без union),
        то почему-то получалось слишком медленно
        """
        class ClassWithFilterMethod:
            def filter(self, *args, **kwargs):
                candidate_responsibles_qs = (
                    CandidateResponsible.objects
                    .filter(*args, **kwargs)
                    .values('candidate_id')
                )

                if issubclass(cls, LastConsiderationBase):
                    kwargs['consideration__is_last'] = True

                consideration_responsibles_qs = (
                    ConsiderationResponsible.objects
                    .filter(*args, **kwargs)
                    .values('consideration__candidate_id')
                )
                return candidate_responsibles_qs.union(consideration_responsibles_qs, all=True)

        return ClassWithFilterMethod()


class ConsiderationResolutionFilterBase(MultipleChoiceFilter):

    model_class = Consideration
    model_value_attr = 'resolution'
    values_field = sform.MultipleChoiceField(
        state=sform.REQUIRED,
        choices=CONSIDERATION_RESOLUTIONS,
    )


@register_filter
class ConsiderationSourceFilter(ConsiderationSourceFilterBase):
    pass


@register_filter
class ConsiderationRecruiterFilter(ConsiderationRecruiterFilterBase):
    pass


@register_filter
class ConsiderationResolutionFilter(ConsiderationResolutionFilterBase):
    pass


# last consideration


class LastConsiderationBase:
    pass


class LastConsiderationMixin(LastConsiderationBase):

    @classmethod
    def get_queryset(cls):
        queryset = super().get_queryset()
        return queryset.filter(is_last=True)


@register_filter
class LastConsiderationExtendedStatusFilter(LastConsiderationMixin, MultipleChoiceFilter):

    condition_classes = [Any, NotAny]
    model_class = Consideration
    model_value_attr = 'extended_status'
    values_field = sform.MultipleChoiceField(
        state=sform.REQUIRED,
        choices=CONSIDERATION_EXTENDED_STATUSES,
    )


@register_filter
class LastConsiderationSourceFilter(LastConsiderationMixin, ConsiderationSourceFilterBase):

    condition_classes = [Any, NotAny]


@register_filter
class LastConsiderationRecruiterFilter(LastConsiderationBase, ConsiderationRecruiterFilterBase):
    pass


@register_filter
class LastConsiderationResolutionFilter(LastConsiderationMixin, ConsiderationResolutionFilterBase):

    condition_classes = [Any, NotAny]


@register_filter
class LastConsiderationExtendedStatusChangedAtFilter(LastConsiderationMixin, DateRangeFilter):

    model_class = Consideration
    date_field = 'extended_status_changed_at'

    @classmethod
    def get_queryset(cls):
        queryset = super().get_queryset()
        return queryset.annotate(
            extended_status_changed_at=Subquery(get_extended_status_changed_at_subquery()[:1]),
        )
