from datetime import date

from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db.models.functions import Lower
from django.urls.base import reverse
from rest_framework import serializers

from intranet.femida.src.candidates.choices import (
    CANDIDATE_DEGREES,
    CONTACT_TYPES,
    LINKED_ENTITY_TYPES,
)
from intranet.femida.src.core.models import City
from intranet.femida.src.core.serializers import MapperField
from intranet.femida.src.hh.models import HHEntityLink
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.skills.models import Skill


CONTACT_TYPE_MAP = {
    'home': CONTACT_TYPES.phone,
    'work': CONTACT_TYPES.phone,
    'cell': CONTACT_TYPES.phone,
    'email': CONTACT_TYPES.email,
    'skype': CONTACT_TYPES.skype,
    'facebook': CONTACT_TYPES.facebook,
    'linkedin': CONTACT_TYPES.linkedin,
}


class HHCandidateContactSerializer(serializers.Serializer):

    type = MapperField(CONTACT_TYPE_MAP, source='type.id')
    account_id = serializers.SerializerMethodField()

    def get_account_id(self, obj):
        if isinstance(obj['value'], dict):
            return obj['value']['formatted']
        return obj['value']


class HHCandidateSocialProfileSerializer(serializers.Serializer):

    type = serializers.SerializerMethodField()
    account_id = serializers.ReadOnlyField(source='url')

    def get_type(self, obj):
        return CONTACT_TYPE_MAP.get(obj['type']['id'], CONTACT_TYPES.other)


class HHCandidateEducationSerializer(serializers.Serializer):

    institution = serializers.CharField(source='name')
    faculty = serializers.CharField(source='organization')
    end_date = serializers.SerializerMethodField()
    degree = serializers.CharField(default=CANDIDATE_DEGREES.unknown)

    def get_end_date(self, obj):
        if not obj.get('year'):
            return None
        return date(obj['year'], 1, 1)


class HHCandidateJobSerializer(serializers.Serializer):

    id = serializers.CharField(default=None)
    employer = serializers.CharField(source='company')
    position = serializers.CharField()
    start_date = serializers.DateField(source='start')
    end_date = serializers.DateField(source='end')
    comment = serializers.CharField(source='description')


class HHCandidateSerializer(serializers.Serializer):

    last_name = serializers.CharField()
    first_name = serializers.CharField()
    middle_name = serializers.CharField()
    birthday = serializers.DateField(source='birth_date')
    gender = serializers.CharField(source='gender.id')

    city = serializers.CharField(source='area.name', default=None)
    target_cities = serializers.SerializerMethodField()

    contacts = serializers.SerializerMethodField()
    candidate_professions = serializers.SerializerMethodField()
    educations = HHCandidateEducationSerializer(source='education.primary', many=True)
    jobs = HHCandidateJobSerializer(source='experience', many=True)

    skills = serializers.SerializerMethodField()
    hh_attachments = serializers.SerializerMethodField()

    source = serializers.CharField(default=SOURCES.external_website)
    source_description = serializers.SerializerMethodField(method_name='_get_hh_url')

    has_missing_fields = serializers.BooleanField()

    def _get_mapped_to_hh(self, model, keys):
        content_type = ContentType.objects.get_for_model(model)
        return set(
            HHEntityLink.objects
            .filter(
                type=LINKED_ENTITY_TYPES.hh,
                content_type=content_type,
                key__in=keys,
            )
            .values_list('object_id', flat=True)
        )

    def get_target_cities(self, obj):
        # https://api.hh.ru/areas
        hh_city_ids = []
        if obj['area']:
            hh_city_ids.append(obj['area']['id'])
        for city in obj['relocation']['area']:
            hh_city_ids.append(city['id'])
        return self._get_mapped_to_hh(City, hh_city_ids)

    def get_candidate_professions(self, obj):
        # https://api.hh.ru/specializations
        salary = '{currency} {amount}'.format(**obj['salary']) if obj['salary'] else ''
        hh_profession_ids = [s['id'] for s in obj['specialization']]
        profession_ids = self._get_mapped_to_hh(Profession, hh_profession_ids)
        return [
            {
                'profession': _id,
                'salary_expectation': salary,
            }
            for _id in profession_ids
        ]

    def get_contacts(self, obj):
        contacts = HHCandidateContactSerializer(obj['contact'], many=True).data
        contacts.extend(HHCandidateSocialProfileSerializer(obj['site'], many=True).data)
        contacts.append({
            'type': CONTACT_TYPES.hh,
            'account_id': self._get_hh_url(obj),
        })
        main_contact_types = ('email', 'phone', 'skype')
        found_main_contact_types = []
        for contact in contacts:
            contact_type = contact['type']
            if contact_type in main_contact_types and contact_type not in found_main_contact_types:
                found_main_contact_types.append(contact_type)
                contact['is_main'] = True
            else:
                contact['is_main'] = False

        return contacts

    def _get_skills_data(self, obj):
        result = [s.lower() for s in obj['skill_set']]
        result.extend(['%s язык' % s['name'].lower() for s in obj['language']])
        return result

    def get_skills(self, obj):
        return set(
            Skill.objects
            .annotate(name_lower=Lower('name'))
            .filter(name_lower__in=self._get_skills_data(obj))
            .values_list('id', flat=True)
        )

    def get_hh_attachments(self, obj):
        name_fields = ('first_name', 'middle_name', 'last_name')
        name = ' '.join(obj[field] for field in name_fields if obj.get(field))
        return [
            {
                'type': _type,
                'url': reverse(
                    viewname='api:hh:resume-file',
                    kwargs={
                        'resume_id': obj['id'],
                        'file_type': _type,
                    },
                ),
                'name': '{}.{}'.format(name, _type),
            }
            for _type in obj['download']
        ]

    def _get_hh_url(self, obj):
        return 'https://{host}/resume/{id}'.format(
            host=settings.HH_HOST,
            id=obj['id'],
        )
