from django.core.management.base import BaseCommand
from django.db.models import Exists, OuterRef

from intranet.femida.src.candidates.choices import (
    CONSIDERATION_STATUSES,
    CONSIDERATION_EXTENDED_STATUSES_WITH_OFFER,
)
from intranet.femida.src.candidates.controllers import (
    bulk_calculate_consideration_extended_statuses,
)
from intranet.femida.src.candidates.models import Consideration
from intranet.femida.src.offers.models import Offer


class Command(BaseCommand):

    help = 'Migrate considerations extended statuses with offers'

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true')
        parser.add_argument('--rollback', action='store_true')

    def handle(self, *args, **options):
        if options['rollback']:
            considerations = (
                Consideration.unsafe
                .filter(extended_status__in=CONSIDERATION_EXTENDED_STATUSES_WITH_OFFER._db_values)
            )
        else:
            considerations = (
                Consideration.unsafe
                .filter(state=CONSIDERATION_STATUSES.in_progress)
                .annotate(
                    is_offer_exist=Exists(
                        Offer.unsafe.alive()
                        .filter(application__consideration=OuterRef('id'))
                        .values('id')
                    ),
                )
                .filter(is_offer_exist=True)
            )

        consideration_extended_statuses = bulk_calculate_consideration_extended_statuses(
            consideration_ids=considerations.values_list('id', flat=True),
            with_offer=not options['rollback'],
        )
        considerations_to_update = []
        for consideration in Consideration.unsafe.filter(id__in=consideration_extended_statuses):
            consideration.extended_status = consideration_extended_statuses[consideration.id]
            considerations_to_update.append(consideration)

        if not options['dry_run']:
            Consideration.unsafe.bulk_update(
                considerations_to_update,
                ['extended_status'],
                batch_size=1000,
            )
        print('Updated ', len(considerations_to_update), ' considerations')
