from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db import transaction, connection
from django.utils.functional import cached_property

from intranet.femida.src.candidates.choices import (
    REFERENCE_STATUSES,
    SUBMISSION_SOURCES,
    SUBMISSION_STATUSES,
)
from intranet.femida.src.candidates.models import Reference, CandidateSubmission
from intranet.femida.src.utils.files import iterate_csv


User = get_user_model()


STATUSES_MAP = {
    'approve': REFERENCE_STATUSES.approved,
    'approve-without-benefits': REFERENCE_STATUSES.approved_without_benefits,
}


class AlreadyExists(Exception):

    pass


class Command(BaseCommand):

    help = 'Create references from csv-sheet'

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true')
        parser.add_argument('--input', '-i', action='store', required=True)

    @cached_property
    def users(self):
        return dict(User.objects.values_list('username', 'id'))

    @cached_property
    def existing_reference_keys(self):
        return set(Reference.objects.values_list('startrek_key', flat=True))

    def _parse(self, row):
        startrek_key = row[1]
        if startrek_key in self.existing_reference_keys:
            raise AlreadyExists
        reference = Reference(
            startrek_key=startrek_key,
            created_by_id=self.users[row[2]],
            created=row[3],
            status=STATUSES_MAP.get(row[4]),
            expiration_date=row[5],
            processed_by_id=self.users[row[6]],
            processed_at=row[7],
            modified=row[7],
        )
        submission = CandidateSubmission(
            candidate_id=row[0].strip(' /').split('/')[-1],
            source=SUBMISSION_SOURCES.reference,
            status=SUBMISSION_STATUSES.closed,
            responsible_id=reference.processed_by_id,
            closed_at=reference.processed_at,
            created=reference.created,
            modified=reference.modified,
        )
        return reference, submission

    @transaction.atomic
    def handle(self, *args, **options):
        references = []
        submissions = []

        rows = iterate_csv(options['input'])
        headers = next(rows)  # noqa
        for row in rows:
            try:
                reference, submission = self._parse(row)
            except AlreadyExists:
                pass
            except Exception as exc:
                print('Error in row:', row, exc)
            else:
                references.append(reference)
                submissions.append(submission)

        if not references:
            print('Nothing to create')
            return

        if not options['dry_run']:
            Reference.objects.bulk_create(references)
            for s, r in zip(submissions, references):
                s.reference = r
            CandidateSubmission.unsafe.bulk_create(submissions)
            submission_ids = ','.join(str(s.id) for s in submissions)

            sql = """
                UPDATE candidates_candidatesubmission as s
                SET
                  first_name = c.first_name,
                  last_name = c.last_name
                FROM candidates_candidate as c
                WHERE s.candidate_id = c.id AND s.id IN (%s);
            """ % submission_ids

            with connection.cursor() as cursor:
                cursor.execute(sql)

        print(len(references), 'references were created')
