from django.core.management.base import BaseCommand
from django.db import transaction

from intranet.femida.src.actionlog.models import Snapshot, SNAPSHOT_REASONS
from intranet.femida.src.candidates.choices import CANDIDATE_STATUSES
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.controllers import update_instance


class Command(BaseCommand):

    help = 'Reopen candidate that was partially closed on offer close'

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true')
        parser.add_argument('--candidate', '-c', action='store', required=True)

    def handle(self, *args, **options):
        dry_run = options['dry_run']
        candidate = Candidate.unsafe.get(id=options['candidate'])
        snapshots = self._find_offer_close_snapshots(candidate)

        if any(s.obj_str == 'consideration' for s in snapshots):
            self.stderr.write('Candidate was closed correctly')
        else:
            self._reopen(candidate, snapshots, dry_run)
            self.stdout.write('Candidate successfully restored')

    @transaction.atomic
    def _reopen(self, candidate, snapshots, dry_run=False):
        responsibles = []
        for snapshot in snapshots:
            if snapshot.obj_str == 'candidate_responsibles':
                responsibles.append(snapshot.data['user'])

        if not dry_run:
            candidate.responsibles.add(*responsibles)
        self.stdout.write('%d responsibles were restored' % len(responsibles))

        source, source_description = self._find_source(candidate)
        if not dry_run:
            update_instance(candidate, data={
                'status': CANDIDATE_STATUSES.in_progress,
                'talent_pool_rate': None,
                'source': source,
                'source_description': source_description,
            })
        self.stdout.write('Source of candidate was restored: %s, %s' % (source, source_description))

    def _find_offer_close_snapshots(self, candidate):
        snapshot = (
            Snapshot.objects
            .filter(
                obj_str='candidate',
                obj_id=candidate.id,
                log_record__action_name='offer_close',
            )
            .prefetch_related('log_record__snapshots')
            .order_by('log_record__action_time')
            .last()
        )
        if not snapshot:
            raise Exception('Could not find "offer_close" event for candidate')
        return snapshot.log_record.snapshots.all()

    def _find_source(self, candidate):
        """
        Находит по логу последнего изменения кандидата его источник
        :return: source, source_description
        """
        snapshot = (
            Snapshot.objects
            .filter(
                obj_str='candidate',
                obj_id=candidate.id,
                reason__in=(
                    SNAPSHOT_REASONS.addition,
                    SNAPSHOT_REASONS.change,
                ),
            )
            .order_by('log_record__action_time')
            .last()
        )
        if not snapshot:
            raise Exception('Could not find source for candidate')
        return snapshot.data.get('source', ''), snapshot.data.get('source_description', '')
