from django.db import models
from django.db.models import Q, OuterRef, Exists
from django.utils import timezone

from intranet.femida.src.candidates.choices import (
    CHALLENGE_TYPES,
    CHALLENGE_STATUSES,
    REFERENCE_STATUSES,
    VERIFICATION_STATUSES,
    VERIFICATION_RESOLUTIONS,
)
from intranet.femida.src.core.db import OrderExpression
from intranet.femida.src.utils.datetime import shifted_now


class CandidateAliveQuerySet(models.QuerySet):

    def touch(self):
        return self.update(touched_at=timezone.now())

    def alive(self):
        return self.filter(is_duplicate=False)


class ChallengeAliveQuerySet(models.QuerySet):

    def alive(self):
        return self.exclude(status=CHALLENGE_STATUSES.cancelled)

    def onedayoffer(self):
        return self.filter(
            type=CHALLENGE_TYPES.contest,
            submission__forms_data__params__passcode__isnull=False,
            submission__forms_data__params__login__isnull=True,
        )


class ReferenceQuerySet(models.QuerySet):

    def alive(self):
        from intranet.femida.src.candidates.models import Consideration

        qs = self.annotate(
            has_prolongation=Exists(
                Consideration.unsafe
                .filter(
                    Q(finished__isnull=True)
                    # FEMIDA-5389 Пролонгируем рекомендацию до выполнения finish_references_task.
                    # 5 дней = кандидата вывели утром ПТ, в вечером ПТ таска запустилась и упала,
                    # в ВТ дежурный ее перезапустил
                    | Q(finished__gte=shifted_now(days=-5)),
                    candidate=OuterRef('submission__candidate_id'),
                    created__lt=OuterRef('expiration_date'),
                )
            ),
        )
        return qs.filter(
            Q(expiration_date__gt=timezone.now()) | Q(has_prolongation=True),
            status__in=(
                REFERENCE_STATUSES.approved,
                REFERENCE_STATUSES.approved_without_benefits,
            ),
        )


class VerificationQuerySet(models.QuerySet):

    def alive(self):
        """
        Актуальные проверки на КИ.

        Проверка считается актуальной, если выполняется любое из условий:
        1. кандидат ещё не заполнил форму, но ссылка ещё актуальна;
        2. кандидат уже заполнил форму, но проверка ещё идёт;
        3. проверка закончена и актуальна на сегодня;
        """
        now = timezone.now()
        is_pending_q = Q(
            status=VERIFICATION_STATUSES.new,
            link_expiration_date__gt=now,
        )
        is_on_check_q = Q(status__in=(
            VERIFICATION_STATUSES.on_check,
            VERIFICATION_STATUSES.on_ess_check,
        ))
        is_actual_q = Q(
            status=VERIFICATION_STATUSES.closed,
            expiration_date__gt=now,
        )
        qs = self.filter(
            is_pending_q
            | is_on_check_q
            | is_actual_q
        )
        qs = qs.annotate(
            status_order=OrderExpression('status', VERIFICATION_STATUSES),
            resolution_order=OrderExpression('resolution', VERIFICATION_RESOLUTIONS),
        )
        return qs.order_by('status_order', '-resolution_order')


ChallengeManager = models.Manager.from_queryset(ChallengeAliveQuerySet)
CandidateManager = models.Manager.from_queryset(CandidateAliveQuerySet)
ReferenceManager = models.Manager.from_queryset(ReferenceQuerySet)
VerificationManager = models.Manager.from_queryset(VerificationQuerySet)
