from logging import getLogger
import re

from django.db import migrations


logger = getLogger(__name__)

INN_KEYS = (
    'Номер ИНН',
    'TIN (Taxpayer Identification Number)',
)
SNILS_KEYS = (
    'Номер страхового свидетельства (ПФР)',
    'Insurance certificate number (Pension Fund of the Russian Federation)',
)
FALSE_ANSWER = 'Нет ответа'

QUESTIONS_RGX = re.compile(
    r"""
    \*\*(.*?)\*\*  # question
    \n
    %%(.*?)%%      # answer
    """,
    re.DOTALL | re.VERBOSE,
)


def parse_forms_wf_questions(raw_questions):
    """
    Основано на intranet.femida.src.utils.forms_constructor.parse_forms_wf_questions
    """
    questions = QUESTIONS_RGX.findall(raw_questions)
    # Если есть дублирующиеся названия вопросов, примется последний из них
    # С инн и снилс таких быть не должно
    return {
        q.strip(): a.strip()
        for q, a in questions
    }


def get_document_id(questions_dict, document_name_translations):
    for question_name in document_name_translations:
        if question_name in questions_dict:
            answer = questions_dict[question_name]
            return answer if answer != FALSE_ANSWER else ''


def migrate_document_ids(apps, schema_editor):
    Verification = apps.get_model('candidates', 'Verification')
    verifications = Verification.objects.exclude(raw_data={})

    migrated = []
    no_raw_questions = []
    parsing_failures = []
    inn_or_snils_not_found = []

    for v in verifications:
        raw_questions = v.raw_data.get('params', {}).get('verification_questions')
        if not raw_questions:
            no_raw_questions.append(v.id)
            continue
        questions_dict = parse_forms_wf_questions(raw_questions)
        if not questions_dict:
            parsing_failures.append(v.id)
            continue

        inn = get_document_id(questions_dict, INN_KEYS)
        snils = get_document_id(questions_dict, SNILS_KEYS)
        if inn is None or snils is None:
            inn_or_snils_not_found.append(v.id)
            continue
        v.raw_data['params']['inn'] = inn
        v.raw_data['params']['snils'] = snils
        v.save(update_fields=['raw_data'])
        migrated.append(v.id)
    logger.info('Migrated: %s', migrated)
    if no_raw_questions or parsing_failures or inn_or_snils_not_found:
        logger.warning('Can not migrate:')
        logger.warning('no_raw_questions: %s', no_raw_questions)
        logger.warning('parsing_failures: %s', parsing_failures)
        logger.warning('inn_or_snils_not_found: %s', inn_or_snils_not_found)


class Migration(migrations.Migration):

    dependencies = [
        ('candidates', '0015_verification_sent_on_check'),
    ]

    operations = [
        migrations.RunPython(migrate_document_ids, reverse_code=lambda *x, **y: None),
    ]
