# Generated by Django 2.2.17 on 2020-12-25 13:18
from collections import defaultdict
from itertools import chain

from django.db import migrations
from django.db.models import F

from intranet.femida.src.candidates.choices import CONTACT_TYPES
from intranet.femida.src.candidates.contacts import normalize_contact_url
from intranet.femida.src.utils.queryset import queryset_iterator


def get_hh_contacts_by_candidate_id(apps):
    CandidateContact = apps.get_model('candidates', 'CandidateContact')
    hh_contacts = (
        CandidateContact.objects
        .filter(
            type=CONTACT_TYPES.hh,
            is_active=True,
        )
        .exclude(normalized_account_id='')
        .values('candidate_id', 'normalized_account_id')
    )
    result = defaultdict(set)

    for contact in hh_contacts:
        result[contact['candidate_id']].add(contact['normalized_account_id'])

    return result


def create_new_hh_contacts_from_source_description(apps, data):
    CandidateContact = apps.get_model('candidates', 'CandidateContact')
    hh_contacts_by_candidate_id = get_hh_contacts_by_candidate_id(apps)

    hh_contacts = []
    for row in data:
        candidate_id = row.candidate_id
        normalized = normalize_contact_url(CONTACT_TYPES.hh, row.source_description)
        if not normalized or normalized in hh_contacts_by_candidate_id[candidate_id]:
            continue

        hh_contacts_by_candidate_id[candidate_id].add(normalized)
        hh_contacts.append(
            CandidateContact(
                account_id=normalized,
                normalized_account_id=normalized,
                candidate_id=candidate_id,
                type=CONTACT_TYPES.hh,
                is_active=True,
            )
        )

    CandidateContact.objects.bulk_create(hh_contacts, batch_size=1000)


def create_hh_contacts(apps, schema_editor):
    Candidate = apps.get_model('candidates', 'Candidate')
    Consideration = apps.get_model('candidates', 'Consideration')

    candidates_data = queryset_iterator(
        queryset=(
            Candidate.unsafe
            .filter(source_description__icontains='resume')
            .annotate(candidate_id=F('id'))
            .values_list('pk', 'candidate_id', 'source_description', named=True)
        ),
        chunk_size=10000,
    )
    considerations_data = queryset_iterator(
        queryset=(
            Consideration.unsafe
            .filter(source_description__icontains='resume')
            .values_list('pk', 'candidate_id', 'source_description', named=True)
        ),
        chunk_size=10000,
    )

    data = chain(candidates_data, considerations_data)
    create_new_hh_contacts_from_source_description(apps, data)


class Migration(migrations.Migration):

    dependencies = [
        ('candidates', '0022_main_recruiter_data_migration'),
    ]

    operations = [
        migrations.RunSQL(
            sql="""
            UPDATE candidates_candidatecontact
            SET normalized_account_id = COALESCE(
                  'https://hh.ru/resume/' || substring(account_id, '.*/resume/([a-zA-Z0-9]+)/?'),
                  normalized_account_id
                )
            WHERE
              type = 'hh'
              AND is_active = 't'
            """,
            reverse_sql=None,
        ),
        migrations.RunPython(
            code=create_hh_contacts,
            reverse_code=lambda *x, **y: None,
        ),
    ]
