import functools
import logging
import re
import struct
import uuid
from collections import defaultdict

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.fields import GenericRelation
from django.contrib.postgres.fields import ArrayField, JSONField
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.utils import timezone
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _
from model_utils.models import TimeStampedModel
from model_utils.fields import AutoLastModifiedField, AutoCreatedField

from intranet.femida.src.attachments.models import Attachment
from intranet.femida.src.candidates.contacts import normalize_contact
from intranet.femida.src.contest.models import Contest, ContestParticipant
from intranet.femida.src.core.choices import GENDER_CHOICES
from intranet.femida.src.core.db.fields import StartrekIssueKeyField, RelativeDeltaField
from intranet.femida.src.core.models import (
    I18NNameModelMixin,
    LanguageTag,
    Currency,
)
from intranet.femida.src.oebs.choices import LOGIN_LOOKUP_DOCUMENT_TYPES
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.permissions.managers.candidate import CandidatePermManager
from intranet.femida.src.permissions.managers.submission import SubmissionPermManager
from intranet.femida.src.permissions.managers.duplication_case import DuplicationCasePermManager
from intranet.femida.src.utils.datetime import shifted_now
from intranet.femida.src.utils.forms_constructor import parse_forms_wf_questions
from intranet.femida.src.wf.models import WFModelMixin

from . import choices
from .managers import CandidateManager, ChallengeManager, ReferenceManager, VerificationManager


logger = logging.getLogger(__name__)


class HistoricalCandidateFields(models.Model):
    """
    Данные, которые снепшотятся в рассмотрение после его завершения
    """
    source = models.CharField(
        max_length=32,
        choices=SOURCES,
        default=SOURCES.other,
        blank=True,
    )
    source_description = models.CharField(
        max_length=255,
        blank=True,
    )

    @cached_property
    def responsibles_by_role(self):
        result = defaultdict(list)
        model_name = self._meta.model_name
        responsibles_attr = f'{model_name}_responsibles'
        responsibles_related_manager = getattr(self, responsibles_attr)
        if responsibles_attr in getattr(self, '_prefetched_objects_cache', {}):
            responsibles = responsibles_related_manager.all()
        else:
            responsibles = responsibles_related_manager.select_related('user')
        for responsible in responsibles:
            result[responsible.role].append(responsible.user)
        return result

    @property
    def recruiters(self):
        return self.responsibles_by_role[choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter]

    @property
    def main_recruiter(self):
        role = choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter
        main_recruiters = self.responsibles_by_role[role]
        return main_recruiters[0] if main_recruiters else None

    # FEMIDA-4914: Поле больше не используется, но в БД пока оставляем
    talent_pool_rate = models.IntegerField(blank=True, null=True)

    class Meta:
        abstract = True


class Candidate(HistoricalCandidateFields, TimeStampedModel):

    unsafe = CandidateManager()
    objects = CandidatePermManager()

    modified = AutoLastModifiedField(_('modified'), db_index=True)

    # TODO: поле неактуальное – надо выпилить
    full_name = models.CharField(
        max_length=255,
        default='',
        blank=True,
    )
    login = models.CharField(
        max_length=64,
        default='',
        blank=True,
        db_index=True,
    )
    startrek_key = StartrekIssueKeyField(
        null=True,
        blank=True,
    )

    first_name = models.CharField(max_length=255)
    middle_name = models.CharField(max_length=255, blank=True)
    last_name = models.CharField(max_length=255)
    birthday = models.DateField(null=True, blank=True)
    gender = models.CharField(
        max_length=6,
        blank=True,
        null=True,
        choices=GENDER_CHOICES,
    )
    country = models.CharField(max_length=255, blank=True)  # потом ссылка на сущность
    city = models.CharField(max_length=255, blank=True)  # потом ссылка на сущность
    target_cities = models.ManyToManyField(
        to='core.City',
        through='candidates.CandidateCity',
        related_name='candidates',
    )
    skills = models.ManyToManyField(
        to='skills.Skill',
        through='candidates.CandidateSkill',
        related_name='candidates',
    )
    attachments = models.ManyToManyField(
        to='attachments.Attachment',
        through='candidates.CandidateAttachment',
        related_name='candidates',
    )
    tags = models.ManyToManyField(
        to='core.Tag',
        through='candidates.CandidateTag',
        related_name='candidates',
    )
    responsibles = models.ManyToManyField(
        to=settings.AUTH_USER_MODEL,
        through='candidates.CandidateResponsible',
        blank=True,
    )
    status = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_STATUSES,
        default=choices.CANDIDATE_STATUSES.in_progress,
    )
    is_from_submission = models.BooleanField(default=True)
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        null=True,
        blank=True,
        related_name='+',
    )
    is_duplicate = models.BooleanField(default=False)
    original = models.ForeignKey(
        to='candidates.Candidate',
        on_delete=models.CASCADE,
        related_name='duplicates',
        blank=True,
        null=True,
    )
    is_hidden = models.BooleanField(default=False)

    oebs_person_id = models.IntegerField(blank=True, null=True)
    is_locked = models.BooleanField(default=False)

    touched_at = models.DateTimeField(
        null=True,
        db_index=True,
        help_text=(
            'Время, когда в последний раз что-то происходило с кандидатом, '
            'а также со связанными с ним значимыми объектами (контакты, скиллы и т.д.). '
            'Это время важно знать для различных синхронизаций с другими системами'
        ),
    )
    ah_modified_at = models.DateTimeField(blank=True, null=True)
    beamery_id = models.UUIDField(blank=True, null=True, unique=True)

    vacancies_mailing_agreement = models.BooleanField(blank=True, null=True)
    events_mailing_agreement = models.BooleanField(blank=True, null=True)

    inn = models.CharField(blank=True, max_length=12, null=True)

    @property
    def first_and_last_names(self):
        partials = self.full_name.split()
        if len(partials) > 1:
            return partials[0], partials[1]
        return ('', '')

    def get_full_name(self):
        return '{} {}'.format(self.first_name, self.last_name)

    @cached_property
    def user(self):
        """
        TODO: Чуть позже есть смысл сделать OneToOne поле c User
        """
        if not self.login:
            return
        return get_user_model().objects.filter(username=self.login).first()

    @property
    def last_consideration(self):
        return self.considerations.filter(is_last=True).last()

    @property
    def main_language(self):
        language_tags = self.candidate_language_tags.filter(is_main=True)
        return LanguageTag.objects.filter(candidate_language_tags__in=language_tags).first()

    @property
    def spoken_languages(self):
        language_tags = self.candidate_language_tags.filter(is_main=False)
        return LanguageTag.objects.filter(candidate_language_tags__in=language_tags)

    def __str__(self):
        return 'Candidate {}: {} {}'.format(self.id, self.first_name, self.last_name)

    class Meta:
        default_manager_name = 'unsafe'
        indexes = [
            models.Index(
                fields=['id', 'touched_at'],
                name='candidate_id_touched_at_idx',
            ),
        ]


class Consideration(HistoricalCandidateFields, TimeStampedModel):

    unsafe = models.Manager()
    objects = CandidatePermManager(perm_prefix='candidate')

    # TODO: Перенести в choices.py
    STATES = choices.CONSIDERATION_STATUSES

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._old_extended_status = self.extended_status

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='considerations',
    )
    # TODO: Переименовать в status
    state = models.CharField(
        max_length=32,
        choices=choices.CONSIDERATION_STATUSES,
        default=choices.CONSIDERATION_STATUSES.in_progress,
    )
    # Расширенный статус рассмотрения.
    # TODO: Возможно, есть смысл объединить с текущим state
    extended_status = models.CharField(
        max_length=32,
        choices=choices.CONSIDERATION_EXTENDED_STATUSES,
        default=choices.CONSIDERATION_EXTENDED_STATUSES.in_progress,
        null=True,
        blank=True,
    )
    resolution = models.CharField(
        max_length=64,
        choices=choices.CONSIDERATION_RESOLUTIONS,
        default='',
        blank=True,
    )
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='created_considerations',
        null=True,
        blank=True,
    )
    is_rotation = models.NullBooleanField()
    rotation = models.OneToOneField(
        to='candidates.Rotation',
        related_name='consideration',
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )
    responsibles = models.ManyToManyField(
        to=settings.AUTH_USER_MODEL,
        through='candidates.ConsiderationResponsible',
        blank=True,
    )

    started = models.DateTimeField(default=timezone.now)
    finished = models.DateTimeField(blank=True, null=True)

    is_last = models.NullBooleanField(default=True)

    @property
    def status(self):
        return self.state

    @property
    def estimation_scale_type(self):
        interviews = self.interviews.all()
        interviews = [i for i in interviews if i.is_alive]
        if all(i.is_pro_level_scale for i in interviews):
            return 'pro_level'
        if all(not i.is_pro_level_scale for i in interviews):
            return 'grade'
        return 'mixed'

    def save(self, *args, **kwargs):
        is_creation = self.pk is None
        super().save(*args, **kwargs)
        if is_creation or self.extended_status != self._old_extended_status:
            ConsiderationHistory.create_from_consideration(self)
            self._old_extended_status = self.extended_status

    def __repr__(self):
        return 'Consideration {}'.format(self.id)

    def __str__(self):
        return f'ID {self.id}, Candidate ID: {self.candidate_id}'

    class Meta:
        default_manager_name = 'unsafe'
        constraints = [
            # Может быть только одно активное рассмотрение
            models.UniqueConstraint(
                fields=['candidate', 'state'],
                name='consideration_unique_in_progress',
                condition=models.Q(state=choices.CONSIDERATION_STATUSES.in_progress),
            ),
        ]


class ConsiderationHistory(models.Model):
    """
    Модель, которая хранит историю изменения статусов рассмотрения
    TODO: Возможно, это стоит делать через actionlog, отчасти для этого он и задумывался.
    Хотя с такой таблицей работать, конечно же, проще.
    """
    consideration = models.ForeignKey(
        to=Consideration,
        on_delete=models.CASCADE,
        related_name='consideration_history',
    )
    status = models.CharField(
        max_length=32,
        choices=choices.CONSIDERATION_EXTENDED_STATUSES,
    )
    changed_at = AutoCreatedField()

    def __str__(self):
        return '{}, {}, {}'.format(self.consideration, self.status, self.changed_at)

    @classmethod
    def create_from_consideration(cls, consideration):
        return cls.objects.create(
            consideration=consideration,
            status=consideration.extended_status,
        )


class ConsiderationIssue(TimeStampedModel):

    consideration = models.ForeignKey(
        to=Consideration,
        on_delete=models.CASCADE,
        related_name='consideration_issues',
    )
    type = models.CharField(max_length=100)
    is_resolved = models.BooleanField(default=False)
    resolved_at = models.DateTimeField(null=True, blank=True)
    resolved_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='resolved_consideration_issues',
        null=True,
        blank=True,
    )
    level = models.CharField(max_length=10, choices=choices.CONSIDERATION_ISSUE_LEVELS)
    params = JSONField(null=True, blank=True)

    def __str__(self):
        result = [
            f'ID: {self.id}',
            f'Consideration ID: {self.consideration_id}',
            f'Type: {self.type}',
            f'Level: {self.level}',
        ]
        if self.is_resolved:
            result.append('Resolved')
        return ", ".join(result)


class CandidateContact(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='contacts'
    )
    type = models.CharField(max_length=32, choices=choices.CONTACT_TYPES)
    account_id = models.CharField(max_length=255)
    normalized_account_id = models.CharField(
        max_length=255,
        blank=True,
        default='',
    )
    is_main = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)
    source = models.CharField(
        max_length=32,
        choices=choices.CONTACT_SOURCES,
        default=choices.CONTACT_SOURCES.other,
        blank=True,
    )

    @property
    def is_normalized(self):
        return self.normalized_account_id != ''

    def save(self, *args, **kwargs):
        # Если ignore_normalization, то сохраняем как есть, иначе нормализуем
        ignore_normalization = kwargs.pop('ignore_normalization', False)
        if not ignore_normalization:
            normalized_account_id = normalize_contact(
                contact_type=self.type,
                account_id=self.account_id,
            )
            self.normalized_account_id = normalized_account_id or ''

        if kwargs.get('update_fields') is not None and 'account_id' in kwargs['update_fields']:
            kwargs['update_fields'].append('normalized_account_id')

        super().save(*args, **kwargs)

    def __str__(self):
        return '{}, {}, {}'.format(self.candidate, self.type, self.account_id)


class CandidateEducation(TimeStampedModel):
    """
    Прошлые места учебы
    """
    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='educations'
    )
    institution = models.CharField(max_length=255)
    faculty = models.CharField(max_length=255, blank=True)
    degree = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_DEGREES,
        default=choices.CANDIDATE_DEGREES.unknown,
    )
    start_date = models.DateField(null=True, blank=True)
    end_date = models.DateField(null=True, blank=True)
    comment = models.TextField(blank=True)

    class Meta:
        indexes = [
            models.Index(
                fields=['candidate', '-end_date'],
                name='edu_candidate_id_end_date_idx',
            ),
        ]

    def __str__(self):
        return '{}, {}'.format(self.candidate, self.institution)


class CandidateJob(TimeStampedModel):
    """
    Прошлые места работы
    """
    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='jobs'
    )
    employer = models.CharField(max_length=255)
    position = models.CharField(max_length=255, blank=True)
    start_date = models.DateField(null=True, blank=True)
    end_date = models.DateField(null=True, blank=True)
    salary_evaluation = models.CharField(max_length=255, blank=True)
    currency = models.CharField(max_length=3, blank=True)
    comment = models.TextField(blank=True)

    def __str__(self):
        return '{}, {}, {}'.format(self.candidate, self.employer, self.position)

    class Meta:
        indexes = [
            models.Index(
                fields=['candidate', '-end_date'],
                name='job_candidate_id_end_date_idx',
            ),
        ]


class CandidateAttachment(TimeStampedModel):

    unsafe = models.Manager()
    objects = CandidatePermManager(perm_prefix='candidate')

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='candidate_attachments',
    )
    # сохраним туда данные из oebs на всякий случай.
    application = models.ForeignKey(
        to='interviews.Application',
        related_name='candidate_attachments',
        on_delete=models.PROTECT,
        db_constraint=False,
        null=True,
        blank=True,
    )
    type = models.CharField(max_length=255, choices=choices.ATTACHMENT_TYPES)
    attachment = models.ForeignKey(
        to=Attachment,
        on_delete=models.PROTECT,
        related_name='candidate_attachments',
    )

    def __str__(self):
        return 'Candidate attachment ({}, {})'.format(
            self.candidate_id, self.attachment_id)

    class Meta:
        default_manager_name = 'unsafe'


class CandidateSkill(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='candidate_skills',
    )
    skill = models.ForeignKey(
        to='skills.Skill',
        on_delete=models.PROTECT,
        related_name='candidate_skills',
    )
    confirmed_by = ArrayField(models.CharField(max_length=255), blank=True)

    def __str__(self):
        return '{}, {}'.format(self.candidate, self.skill)

    class Meta:
        unique_together = ('candidate', 'skill')


class CandidateTag(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='candidate_tags',
    )
    tag = models.ForeignKey(
        to='core.Tag',
        on_delete=models.PROTECT,
        related_name='candidate_tags',
    )
    is_active = models.BooleanField(default=True, null=True)

    def __str__(self):
        return '{}, Tag: {}'.format(self.candidate, self.tag)

    class Meta:
        unique_together = ('candidate', 'tag')


class CandidateSubmission(WFModelMixin, TimeStampedModel):
    """
    Отклик будущего кандидата на вакансии Яндекса
    """
    unsafe = models.Manager()
    objects = SubmissionPermManager()

    WIKI_FIELDS_MAP = {
        'comment': 'formatted_comment',
    }

    # Данные с формы создания (отклика / рекомендации)
    first_name = models.CharField(max_length=255)
    last_name = models.CharField(max_length=255)
    email = models.CharField(max_length=255, blank=True, null=True)
    phone = models.CharField(max_length=64, blank=True, null=True)
    comment = models.TextField(blank=True, null=True)
    formatted_comment = models.TextField(blank=True, null=True)
    target_cities = models.ManyToManyField(
        to='core.City',
        related_name='submissions',
        blank=True,
    )
    skills = models.ManyToManyField(
        to='skills.Skill',
        related_name='submissions',
        blank=True,
    )
    professions = models.ManyToManyField(
        to='professions.Profession',
        related_name='submissions',
        blank=True,
    )
    attachment = models.ForeignKey(
        to='attachments.Attachment',
        on_delete=models.PROTECT,
        related_name='submissions',
        null=True,
        blank=True,
    )

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.PROTECT,
        related_name='submissions',
        null=True,
        blank=True,
    )
    source = models.CharField(
        max_length=32,
        choices=choices.SUBMISSION_SOURCES,
    )
    publication = models.ForeignKey(
        to='publications.Publication',
        related_name='submissions',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    form = models.ForeignKey(
        to='vacancies.SubmissionForm',
        related_name='submissions',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    reference = models.OneToOneField(
        to='candidates.Reference',
        related_name='submission',
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )
    rotation = models.OneToOneField(
        to='candidates.Rotation',
        related_name='submission',
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )
    # Ссылка на форму, с которой пришел запрос в КФ
    publication_url = models.CharField(
        max_length=2048,
        default='',
        blank=True,
        null=True,
    )
    # Поле нужно для истории, чтобы сохранить список вакансий на момент закрытия отклика
    vacancies = models.ManyToManyField(
        to='vacancies.Vacancy',
        related_name='submissions',
        blank=True,
    )
    status = models.CharField(
        max_length=32,
        choices=choices.SUBMISSION_STATUSES,
        default=choices.SUBMISSION_STATUSES.new,
    )
    responsible = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='submissions_responsible_for',
        null=True,
        blank=True,
    )
    forms_data = JSONField(null=True, blank=True)
    # Поле используется только в откликах, созданных после релиза FEMIDA-5050,
    # т.к. в исторических данных много дублирующихся откликов
    forms_answer_id = models.CharField(
        max_length=255,
        blank=True,
        null=True,
        unique=True,
    )

    closed_at = models.DateTimeField(null=True, blank=True)

    # Признак быстрого отказа
    is_fast_rejection = models.NullBooleanField(default=False)

    # TODO: Это property сейчас нужно для обратной совместимости.
    # Нужно постепенно от него избавиться.
    @cached_property
    def candidate_data(self):
        if self.source in choices.EXTERNAL_SUBMISSION_SOURCES:
            return self.forms_data.get('params', {})
        else:
            return {
                'cand_name': self.first_name,
                'cand_surname': self.last_name,
                'cand_phone': self.phone,
                'cand_email': self.email,
                'cand_info': self.comment,
            }

    @property
    def raw_questions(self):
        """
        Список технических вопросов с формы (могут отсутствовать).
        Используются для автоматического создания challenge с ответами на них.
        """
        return self.candidate_data.get('cand_questions')

    @cached_property
    def parsed_questions(self):
        pattern = re.compile(
            r"""
            \*\*(.*?)\*\*  # question
            \n
            %%(.*?)%%      # answer
            """,
            re.DOTALL | re.VERBOSE,
        )
        return parse_forms_wf_questions(self.raw_questions, pattern)

    @property
    def form_cv_url(self):
        """
        URL на файл в MDS КФ
        """
        return self.candidate_data.get('cand_cv')

    @property
    def yuid(self):
        return self.candidate_data.get('cand_yuid')

    @property
    def form_url(self):
        return self.candidate_data.get('form_url')

    @property
    def login(self):
        return self.candidate_data.get('login')

    @property
    def contest_id(self):
        contest_id = self.candidate_data.get('contest_id')
        if contest_id:
            try:
                return int(contest_id)
            except ValueError:
                logger.warning('contest_id is not an integer for Submission %d', self.id)

    @property
    def is_contest_notification_required(self):
        """
        Нужно ли отправлять кандидату письмо при регистрации на contest
        """
        if self.candidate_data.get('notify', '').lower() == 'false':
            return False
        return True

    @property
    def is_internship(self):
        return (
            self.source == choices.SUBMISSION_SOURCES.form and self.form.is_internship
            or (
                self.source == choices.SUBMISSION_SOURCES.publication
                and self.publication.is_internship
            )
        )

    @property
    def passcode(self):
        return self.candidate_data.get('passcode')

    @property
    def is_onedayoffer(self):
        return self.passcode and not self.login

    @property
    def answer_id(self):
        return self.candidate_data.get('answer_id')

    @property
    def autoestimate(self):
        """
        FEMIDA-6551: Флаг автоматического завершения испытания для ОДО

        """
        if self.candidate_data.get('autoestimate', '').lower() == 'false':
            return False
        return True

    def __str__(self):
        return 'ID: {submission_id}, Form ID: {form_id}, Candidate ID: {candidate_id}'.format(
            submission_id=self.id,
            form_id=self.form_id,
            candidate_id=self.candidate_id,
        )

    class Meta:
        default_manager_name = 'unsafe'


class Reference(TimeStampedModel):

    objects = ReferenceManager()

    vacancies = models.ManyToManyField(
        to='vacancies.Vacancy',
        related_name='references',
        blank=True,
    )
    startrek_key = StartrekIssueKeyField(null=True, blank=True)
    status = models.CharField(
        max_length=32,
        choices=choices.REFERENCE_STATUSES,
        default=choices.REFERENCE_STATUSES.new,
    )
    expiration_date = models.DateTimeField(
        default=functools.partial(shifted_now, **settings.REFERENCE_LIFETIME),
    )
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='created_references',
    )

    # Заполняются в момент обработки рекомендации координатором
    processed_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='processed_references',
        null=True,
        blank=True,
    )
    processed_at = models.DateTimeField(null=True, blank=True)

    @property
    def is_approved(self):
        return self.status in (
            choices.REFERENCE_STATUSES.approved,
            choices.REFERENCE_STATUSES.approved_without_benefits,
        )

    @property
    def lifetime(self):
        return self.expiration_date - self.created

    def __str__(self):
        try:
            submission_id = self.submission.id
        except ObjectDoesNotExist:
            submission_id = None

        return 'Reference {} [{}] on Submission {}'.format(
            self.id,
            self.status,
            submission_id,
        )


class Rotation(TimeStampedModel):

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='created_rotations',
    )
    vacancies = models.ManyToManyField(
        to='vacancies.Vacancy',
        related_name='rotations',
    )
    startrek_rotation_key = StartrekIssueKeyField()
    startrek_myrotation_key = StartrekIssueKeyField()
    status = models.CharField(
        max_length=32,
        choices=choices.ROTATION_STATUSES,
        default=choices.ROTATION_STATUSES.new,
    )

    def __str__(self):
        try:
            submission_id = self.submission.id
        except ObjectDoesNotExist:
            submission_id = None

        return 'Rotation {} [{}] on Submission {}'.format(
            self.id,
            self.status,
            submission_id,
        )

    class Meta:
        constraints = [
            # У одного сотрудника может быть только одна необработанная ротация
            models.UniqueConstraint(
                fields=['created_by', 'status'],
                name='rotation_unique_status_new',
                condition=models.Q(status=choices.ROTATION_STATUSES.new),
            ),
        ]


class CryptaVector(TimeStampedModel):
    """
    В этой таблице хранятся вектора пользователей (yandexuid), полученные из векторов сайтов,
    которые посещал этот пользователь.

    https://st.yandex-team.ru/FEMIDA-4318

    Раньше вектора сайтов считались по логам Браузера (vector_b) и логам Метрики (vector_m).
    Теперь вектора сайтов считаются только по логам Браузера (vector).

    Вектора (vector_b, vector_m) и (vector) посчитаны в разных пространствах, поэтому их
    нельзя использовать вместе.
    """
    submission = models.ForeignKey(
        to=CandidateSubmission,
        on_delete=models.PROTECT,
        related_name='crypta_vectors',
    )
    yandexuid = models.CharField(max_length=20)
    vector_b = models.BinaryField(null=True)  # Вектор по логам Браузера размерности 256
    vector_m = models.BinaryField(null=True)  # Вектор по логам Метрики размерности 256
    vector = models.BinaryField(null=True)  # Вектор по логам Браузера размерности 512

    def _parse_vector(self, vector, size):
        if vector:
            return struct.unpack('f' * size, vector)

    @cached_property
    def parsed_vector_b(self):
        return self._parse_vector(self.vector_b, 256)

    @cached_property
    def parsed_vector_m(self):
        return self._parse_vector(self.vector_m, 256)

    @cached_property
    def parsed_vector(self):
        return self._parse_vector(self.vector, 512)

    def __str__(self):
        return 'CryptaVector for Submission {}, yandexuid {}'.format(
            self.submission.id, self.yandexuid
        )


class CandidateProfession(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        related_name='candidate_professions',
        on_delete=models.PROTECT,
    )
    profession = models.ForeignKey(
        to='professions.Profession',
        related_name='candidate_professions',
        on_delete=models.PROTECT,
    )
    professional_sphere = models.ForeignKey(
        to='professions.ProfessionalSphere',
        related_name='candidate_professions',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    salary_expectation = models.CharField(max_length=255, blank=True)

    def __str__(self):
        return '{}, {}'.format(self.candidate, self.profession)

    class Meta:
        unique_together = ('candidate', 'profession')


class CandidateCity(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        related_name='candidate_cities',
        on_delete=models.PROTECT,
    )
    city = models.ForeignKey(
        to='core.City',
        related_name='candidate_cities',
        on_delete=models.PROTECT,
    )

    def __str__(self):
        return '{}, {}'.format(self.candidate, self.city)

    class Meta:
        unique_together = ('candidate', 'city')


class Challenge(WFModelMixin, TimeStampedModel):

    objects = ChallengeManager()

    WIKI_FIELDS_MAP = {
        'comment': 'formatted_comment',
    }

    type = models.CharField(
        max_length=32,
        choices=choices.CHALLENGE_TYPES,
        default=choices.CHALLENGE_TYPES.quiz,
    )
    status = models.CharField(
        max_length=32,
        choices=choices.CHALLENGE_STATUSES,
        default=choices.CHALLENGE_STATUSES.assigned,
    )
    candidate = models.ForeignKey(
        to=Candidate,
        related_name='challenges',
        on_delete=models.PROTECT,
    )
    resolution = models.CharField(
        max_length=32,
        choices=choices.CHALLENGE_RESOLUTIONS,
        default='',
        blank=True,
        null=True,
    )
    comment = models.TextField(blank=True)
    formatted_comment = models.TextField(blank=True)
    reviewed_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        related_name='reviewed_challenges',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    submission = models.ForeignKey(
        to=CandidateSubmission,
        related_name='challenges',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    consideration = models.ForeignKey(
        to=Consideration,
        related_name='challenges',
        on_delete=models.PROTECT,
        null=True,
    )
    application = models.ForeignKey(
        to='interviews.Application',
        related_name='challenges',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    # TODO: Переименовать в data
    answers = JSONField(null=True)

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='created_challenges',
        blank=True,
        null=True,
    )
    finished = models.DateTimeField(
        null=True,
        blank=True,
    )
    finished_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='finished_challenges',
        null=True,
        blank=True,
    )
    cancel_reason = models.TextField(
        blank=True,
    )

    comments = GenericRelation('comments.Comment')

    # здесь могут храниться ключи из разных очередей
    startrek_onedayoffer_key = StartrekIssueKeyField(null=True, blank=True)
    contest = models.ForeignKey(
        to=Contest,
        on_delete=models.PROTECT,
        related_name='contest',
        null=True,
        blank=True
    )
    contest_participant = models.ForeignKey(
        to=ContestParticipant,
        on_delete=models.PROTECT,
        related_name='contest_participant',
        null=True,
        blank=True
    )

    @property
    def participant_id(self):
        if self.type != choices.CHALLENGE_TYPES.contest:
            return None
        return self.answers.get('participation', {}).get('id')


class DuplicationCase(TimeStampedModel):

    unsafe = models.Manager()
    objects = DuplicationCasePermManager()

    first_candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='+',
    )
    second_candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='+',
        blank=True,
        null=True,
    )
    merged_to = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='+',
        blank=True,
        null=True,
    )
    submission = models.ForeignKey(
        to=CandidateSubmission,
        on_delete=models.CASCADE,
        related_name='duplication_cases',
        blank=True,
        null=True,
    )
    # TODO: Удалить поле после миграции статусов
    is_active = models.BooleanField(default=True)
    status = models.CharField(
        max_length=32,
        choices=choices.DUPLICATION_CASE_STATUSES,
        default=choices.DUPLICATION_CASE_STATUSES.new,
    )
    resolution = models.CharField(
        max_length=32,
        choices=choices.DUPLICATION_CASE_RESOLUTIONS,
        blank=True,
        null=True,
    )
    managed_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='duplication_cases',
        blank=True,
        null=True,
    )
    score = models.FloatField(blank=True, null=True)
    # Пометка, считаем ли мы, что это нужно автоматически мержить
    is_auto_merge = models.BooleanField(default=False)

    @property
    def similarity_info(self):
        from intranet.femida.src.candidates.deduplication import SimilarityInfo
        return SimilarityInfo(self.first_candidate, self.second_candidate)

    def __str__(self):
        return 'Case: ({}, {})'.format(
            self.first_candidate_id,
            self.second_candidate_id,
        )

    class Meta:
        default_manager_name = 'unsafe'


class Verification(TimeStampedModel):
    """
    Проверка кандидата на КИ – конфликт интересов
    """
    objects = VerificationManager()

    type = models.CharField(
        max_length=32,
        choices=choices.VERIFICATION_TYPES,
        default=choices.VERIFICATION_TYPES.default,
        blank=False,
    )
    grade = models.PositiveSmallIntegerField(null=True)

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='verifications',
    )
    application = models.ForeignKey(
        to='interviews.Application',
        on_delete=models.PROTECT,
        related_name='verifications',
    )
    uuid = models.UUIDField(default=uuid.uuid4)
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='created_verifications',
    )
    status = models.CharField(
        max_length=32,
        choices=choices.VERIFICATION_STATUSES,
        default=choices.VERIFICATION_STATUSES.new,
        blank=True,
    )
    link_expiration_date = models.DateTimeField(
        default=functools.partial(shifted_now, days=30),
    )
    startrek_ess_key = StartrekIssueKeyField(blank=True)
    raw_data = JSONField(blank=True, default=dict)
    resolution = models.CharField(
        max_length=32,
        choices=choices.VERIFICATION_RESOLUTIONS,
        blank=True,
    )
    expiration_date = models.DateTimeField(
        null=True,
        blank=True,
    )
    sent_on_check = models.DateTimeField(
        null=True,
        blank=True,
    )

    @classmethod
    def get_link(cls, uuid: str, type: choices.VERIFICATION_TYPES):
        verification_type = type or choices.VERIFICATION_TYPES.default
        return 'https://{forms_host}/surveys/{survey_id}/?uuid={uuid}&{params}'.format(
            forms_host=settings.FORMS_EXT_HOST,
            survey_id=settings.VERIFICATION_SETTINGS[verification_type]['id'],
            uuid=uuid,
            params=settings.VERIFICATION_SETTINGS[verification_type]['params']
        )

    @property
    def link(self):
        return self.get_link(self.uuid.hex, self.type)

    @property
    def raw_questions(self):
        """
        Полученный от КФ упорядоченный список пар вопрос-ответ.
        Возможно дублирование текста вопроса.
        """
        params = self.raw_data.get('params', {})
        return params.get('verification_questions')

    @property
    def parsed_questions(self):
        return parse_forms_wf_questions(self.raw_questions)

    @cached_property
    def form_data(self):
        return self.raw_data.get('params', {})

    @cached_property
    def inn(self):
        return self.form_data.get(LOGIN_LOOKUP_DOCUMENT_TYPES.inn, '')

    @cached_property
    def snils(self):
        return self.form_data.get(LOGIN_LOOKUP_DOCUMENT_TYPES.snils, '')

    def __str__(self):
        return 'Verification {} [{}] on Candidate {}'.format(
            self.id,
            self.status,
            self.candidate_id,
        )


class CandidateResponsible(models.Model):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='candidate_responsibles',
    )
    user = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='candidate_responsibles',
    )
    role = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_RESPONSIBLE_ROLES,
        default=choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter,
        null=True,
    )

    class Meta:
        constraints = [
            # Может быть только один основной рекрутер
            models.UniqueConstraint(
                fields=['candidate', 'role'],
                name='candidate_has_single_main_recruiter',
                condition=models.Q(role=choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter),
            ),
        ]
        db_table = 'candidates_candidate_responsibles'
        unique_together = ('candidate', 'user', 'role')


class ConsiderationResponsible(models.Model):

    consideration = models.ForeignKey(
        to=Consideration,
        on_delete=models.CASCADE,
        related_name='consideration_responsibles',
    )
    user = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='consideration_responsibles',
    )
    role = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_RESPONSIBLE_ROLES,
        default=choices.CANDIDATE_RESPONSIBLE_ROLES.recruiter,
        null=True,
    )

    class Meta:
        constraints = [
            # Может быть только один основной рекрутер
            models.UniqueConstraint(
                fields=['consideration', 'role'],
                name='consideration_has_single_main_recruiter',
                condition=models.Q(role=choices.CANDIDATE_RESPONSIBLE_ROLES.main_recruiter),
            ),
        ]
        db_table = 'candidates_consideration_responsibles'
        unique_together = ('consideration', 'user', 'role')


class ScoringCategory(I18NNameModelMixin, models.Model):

    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)
    is_active = models.BooleanField(default=False)

    def __str__(self):
        return f'ScoringCategory {self.id}, {self.name}'


class CandidateScoring(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='candidate_scorings',
    )
    scoring_category = models.ForeignKey(
        to=ScoringCategory,
        on_delete=models.CASCADE,
        related_name='candidate_scorings',
    )
    scoring_value = models.FloatField()
    version = models.CharField(max_length=32)

    def __str__(self):
        return f'CandidateScoring {self.id} on Candidate {self.candidate_id}'

    class Meta:
        unique_together = ('candidate', 'scoring_category', 'version')


class CandidateLanguageTag(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='candidate_language_tags',
    )
    tag = models.ForeignKey(
        to=LanguageTag,
        on_delete=models.CASCADE,
        related_name='candidate_language_tags',
    )
    is_main = models.BooleanField(
        default=False,
    )

    class Meta:
        constraints = [
            # Может быть только один основной язык
            models.UniqueConstraint(
                fields=['candidate', 'is_main'],
                name='candidate_has_only_one_main_language',
                condition=models.Q(is_main=True),
            )
        ]
        unique_together = ['tag', 'candidate']


class CandidateCostsSet(TimeStampedModel):

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='candidate_costs_set',
    )

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
    )


class CandidateCost(models.Model):

    candidate_costs_set = models.ForeignKey(
        to=CandidateCostsSet,
        on_delete=models.CASCADE,
        related_name='costs',
    )

    cost_group = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_COST_GROUPS,
        default=choices.CANDIDATE_COST_GROUPS.expectation,
        null=False,
    )

    type = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_COST_TYPES,
        default=choices.CANDIDATE_COST_TYPES.total,
        null=False,
    )

    value = models.PositiveIntegerField(
        default=None,
        null=True,
    )

    # Отсутствие валюты означает цену в произвольных единицах.
    # Применимо к акциям (шт.), например.
    currency = models.ForeignKey(
        to=Currency,
        on_delete=models.PROTECT,
        default=None,
        null=True,
    )

    rate = models.CharField(
        max_length=32,
        choices=choices.CANDIDATE_COST_RATES,
        default=None,
        null=True,
    )

    taxed = models.BooleanField(
        default=None,
        null=True,
    )

    comment = models.CharField(
        max_length=255,
        default=None,
        null=True,
    )

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=["candidate_costs_set", "cost_group", "type"],
                name="idxccth",
            )]


class CandidateSource(models.Model):

    name = models.CharField(
        max_length=32,
    )
    time_to_live = RelativeDeltaField(
        default=None,
        null=True,
        blank=True,
        help_text=RelativeDeltaField.help_text,
    )
    categories = models.ManyToManyField(
        to='CandidateSourceCategory',
        through='CandidateCompoundSource',
        blank=True,
    )

    def __str__(self):
        return f'CandidateSource {self.pk}: {self.name}'


class CandidateSourceCategory(models.Model):

    name = models.CharField(
        max_length=32,
    )
    sources = models.ManyToManyField(
        to=CandidateSource,
        through='CandidateCompoundSource',
        blank=True,
    )

    def __str__(self):
        return f'CandidateSourceCategory {self.pk}: {self.name}'


class CandidateCompoundSource(models.Model):

    category = models.ForeignKey(
        to=CandidateSourceCategory,
        null=False,
        on_delete=models.CASCADE,
    )

    source = models.ForeignKey(
        to=CandidateSource,
        null=False,
        on_delete=models.CASCADE,
    )

    def __str__(self):
        return f'CandidateCompoundSource {self.pk}: {self.category.name}/{self.source.name}'

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=["category", "source"],
                name="idxccscs",
            )]


class CandidateTouch(TimeStampedModel):

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.SET_NULL,
        null=True,
    )

    source = models.ForeignKey(
        to=CandidateCompoundSource,
        related_name='touches',
        null=False,
        on_delete=models.CASCADE,
    )

    candidate = models.ForeignKey(
        to=Candidate,
        related_name='touches',
        null=False,
        on_delete=models.CASCADE,
    )

    consideration = models.ForeignKey(
        to=Consideration,
        related_name='candidate_touches',
        null=True,
        default=None,
        blank=True,
        on_delete=models.SET_NULL,
    )

    application = models.ForeignKey(
        to='interviews.Application',
        related_name='candidate_touches',
        null=True,
        default=None,
        blank=True,
        on_delete=models.SET_NULL,
    )

    expiration_date = models.DateField(
        null=True,
        default=None,
        blank=True,
    )

    description = models.CharField(
        max_length=1024,
        null=True,
        default=None,
        blank=True,
    )

    medium = models.CharField(
        choices=choices.CANDIDATE_TOUCH_MEDIUMS,
        max_length=32,
    )
