import logging
import random

from django.core.exceptions import ValidationError
from django.db import transaction
from django.template import loader

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.candidates.choices import (
    ROTATION_STATUSES,
    SUBMISSION_SOURCES,
    SUBMISSION_STATUSES,
)
from intranet.femida.src.candidates.models import CandidateSubmission, Rotation
from intranet.femida.src.candidates.startrek.issues import (
    create_rotation_issue,
    create_myrotation_issue,
)
from intranet.femida.src.candidates.submissions.controllers import SubmissionController
from intranet.femida.src.candidates.tasks import (
    create_rotation_issue_task,
    create_myrotation_issue_task,
)
from intranet.femida.src.staff.helpers import get_hr_partners
from intranet.femida.src.startrek.operations import IssueTransitionOperation
from intranet.femida.src.startrek.tasks import add_issue_comment_task
from intranet.femida.src.startrek.utils import (
    TransitionEnum,
    ResolutionEnum,
    StatusEnum,
    StartrekError,
    get_issue,
)


logger = logging.getLogger(__name__)


def create_rotation(data, initiator):
    rotation = Rotation.objects.create(created_by=initiator)
    rotation.vacancies.set(data['publications'])

    CandidateSubmission.objects.create(
        rotation=rotation,
        source=SUBMISSION_SOURCES.rotation,
        status=SUBMISSION_STATUSES.draft,
        first_name=rotation.created_by.first_name,
        last_name=rotation.created_by.last_name,
        comment=data['comment'],
    )

    context = {
        'reason': data['reason'],
        'comment': data['comment'],
        'is_privacy_needed': data['is_privacy_needed'],
    }

    hr_partners = [i.username for i in get_hr_partners(rotation.created_by.department)]
    kwargs = {
        'hr_partners': hr_partners,
        'assignee': random.choice(hr_partners),
    } if hr_partners else {}
    try:
        create_rotation_issue(rotation, context, **kwargs)
        create_myrotation_issue(rotation, context, **kwargs)
    except StartrekError:
        # Если не удалось создать хоть один из тикетов,
        # не разбираемся, а запускаем всю цепочку в фоне
        task_kwargs = dict({
            'rotation_id': rotation.id,
            'context': context,
        }, **kwargs)
        tasks_chain = (
            create_rotation_issue_task.si(**task_kwargs)
            | create_myrotation_issue_task.si(**task_kwargs)
        )
        tasks_chain.delay()

    return rotation


@action_logged('rotation_approve')
@transaction.atomic
def rotation_approve(rotation):
    """
    Подтверждение ротации
    :param rotation: Rotation
    """
    issue = get_issue(rotation.startrek_rotation_key)
    if issue.status.key != StatusEnum.validated:
        logger.warning(
            'Could not approve rotation. Invalid issue %s status: %s',
            issue.key,
            issue.status.key,
        )
        raise ValidationError('rotation_issue_invalid_status')

    rotation.status = ROTATION_STATUSES.approved
    rotation.save(update_fields=('status', 'modified'))

    rotation.submission.status = SUBMISSION_STATUSES.new
    rotation.submission.save(update_fields=('status', 'modified'))

    comment = loader.render_to_string('startrek/rotations/myrotation-approved.wiki')
    IssueTransitionOperation(rotation.startrek_myrotation_key).delay(
        transition=TransitionEnum.validated,
        comment=comment,
    )
    comment = loader.render_to_string('startrek/rotations/rotation-approved.wiki')
    add_issue_comment_task.delay(
        keys=rotation.startrek_rotation_key,
        text=comment,
    )


@action_logged('rotation_reject')
@transaction.atomic
def rotation_reject(rotation):
    """
    Отклонение ротации
    :param rotation: Rotation
    """
    issue = get_issue(rotation.startrek_rotation_key)
    if issue.status.key != StatusEnum.closed:
        logger.warning(
            'Could not reject rotation. Invalid issue %s status: %s',
            issue.key,
            issue.status.key,
        )
        raise ValidationError('rotation_issue_invalid_status')

    rotation.status = ROTATION_STATUSES.rejected
    rotation.save(update_fields=('status', 'modified'))

    controller = SubmissionController(rotation.submission)
    controller.reject()

    context = {
        'hr_partners': get_hr_partners(rotation.created_by.department),
    }
    comment = loader.render_to_string('startrek/rotations/myrotation-rejected.wiki', context)
    IssueTransitionOperation(rotation.startrek_myrotation_key).delay(
        transition=TransitionEnum.closed,
        resolution=ResolutionEnum.wont_fix,
        comment=comment,
    )
