from django.db.models import Q

from intranet.femida.src.candidates.choices import (
    CONSIDERATION_STATUSES,
    ROTATION_STATUSES,
    SUBMISSION_STATUSES,
)
from intranet.femida.src.candidates.models import Consideration, Rotation


def get_current_rotation(user):
    """
    Отдаёт действующую заявку на ротацию, одну из:
    1. Необработанная заявка
    2. Подтверждённая заявка, с ещё необработанным откликом
    3. Подтверждённая заявка, связанная с активным рассмотрением
    """
    new_rotation_q = Q(status=ROTATION_STATUSES.new)
    approved_rotation_q = Q(
        status=ROTATION_STATUSES.approved,
        submission__status=SUBMISSION_STATUSES.new,
    )
    active_rotation_q = Q(consideration__state=CONSIDERATION_STATUSES.in_progress)
    return (
        Rotation.objects
        .filter(
            Q(created_by=user)
            & (
                new_rotation_q
                | approved_rotation_q
                | active_rotation_q
            )
        )
        .first()
    )


def user_has_active_consideration(user):
    """
    У пользователя есть активное рассмотрение
    """
    return (
        Consideration.unsafe
        .filter(
            state=CONSIDERATION_STATUSES.in_progress,
            candidate__login=user.username,
            candidate__is_duplicate=False,
        )
        .exists()
    )


def is_user_rotating(user):
    return (
        user_has_active_consideration(user)
        or get_current_rotation(user) is not None
    )
