from typing import List, Optional

from django.conf import settings
from rest_framework import serializers
import waffle

from intranet.femida.src.candidates.choices import VERIFICATION_TYPES
from intranet.femida.src.candidates.models import Verification
from intranet.femida.src.candidates.verifications.helpers import is_verification_required_for_department
from intranet.femida.src.core.serializers import StartrekUniqueField
from intranet.femida.src.core.switches import TemporarySwitch


def _get_grade_tag_for_ess_issue(verification: Verification) -> str:
    if verification.grade >= 17:
        return 'Grade 17 or higher'
    else:
        return 'Grade 16 or lower'


class EssIssuesFieldsSerializer(serializers.Serializer):
    # Common ESS fields
    unique = StartrekUniqueField(queue=settings.STARTREK_SALARY_QUEUE)
    components = serializers.SerializerMethodField()
    tags = serializers.SerializerMethodField()

    # international_by_grade fields
    candidateCitizenship = serializers.SerializerMethodField()

    def get_components(self, obj: Verification) -> List[str]:
        components = []
        if obj.type == VERIFICATION_TYPES.international_by_grade:
            components.append(settings.STARTREK_ESS_INTERNATIONAL_COMPONENT)
        else:
            components.append(settings.STARTREK_ESS_BGS_COMPONENT)

        return components

    def get_tags(self, obj: Verification) -> List[str]:
        tags = []
        # TODO: убрать свитч после релиза FEMIDA-7226
        if waffle.switch_is_active(TemporarySwitch.NEW_VERIFICATION_FLOW):
            tags.append(settings.STARTREK_ESS_AWAITING_CHECKS_TAG)

        # Note: если анкету КИ отправляли по вакансии в подразделение,
        # для которого проверка КИ не обязательна – ставим специальный тег.
        # На практике он может вообще не отражать действительности, потому что
        # подразделение, в которое выставляют оффер, может отличаться от того,
        # на которое проводили проверку КИ.
        # Но мы тут полагаемся на "ровные процессы",
        # а тег ставим для привлечения внимания к подобным кандидатам
        if not is_verification_required_for_department(obj.application.vacancy.department):
            tags.append(settings.STARTREK_ESS_SKIP_VALIDATION_TAG)

        if obj.type == VERIFICATION_TYPES.international_by_grade and obj.grade is not None:
            tags.append(_get_grade_tag_for_ess_issue(obj))

        return tags

    def get_candidateCitizenship(self, obj: Verification) -> Optional[str]:
        if 'params' in obj.raw_data:
            return obj.raw_data['params'].get('citizenship')
