from django.db.models import Q

from intranet.femida.src.applications.models import Application
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.candidates.choices import (
    CONTACT_TYPES,
    CONTACT_SOURCES,
    SUBMISSION_STATUSES,
    SUBMISSION_SOURCES,
)
from intranet.femida.src.communications.choices import MESSAGE_TYPES
from intranet.femida.src.interviews.choices import APPLICATION_RESOLUTIONS
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.vacancies.choices import VACANCY_ROLES
from intranet.femida.src.vacancies.models import VacancyMembership, Vacancy


def extract_contacts(submission):
    contacts = []
    if submission.phone:
        contacts.append({
            'type': CONTACT_TYPES.phone,
            'account_id': submission.phone,
            'is_main': True,
            'source': CONTACT_SOURCES.forms,
        })
    if submission.email:
        contacts.append({
            'type': CONTACT_TYPES.email,
            'account_id': submission.email,
            'is_main': True,
            'source': CONTACT_SOURCES.forms,
        })
    return contacts


def prepare_candidate_data(submission):
    common_fields = {
        'first_name': submission.first_name,
        'last_name': submission.last_name,
        'contacts': extract_contacts(submission),
        'attachments': [submission.attachment] if submission.attachment else [],
        'source': get_candidate_source_by_submission(submission),
    }
    if submission.source == SUBMISSION_SOURCES.reference:
        return dict(common_fields, **{
            'skills': submission.skills.all(),
            'target_cities': submission.target_cities.all(),
            'new_professions': submission.professions.all(),
        })
    elif submission.source == SUBMISSION_SOURCES.rotation:
        return dict(common_fields, **{
            'login': submission.rotation.created_by.username,
        })
    return common_fields


def prepare_message_data(candidate, submission):
    return {
        'candidate': candidate,
        'text': submission.comment,
        'type': MESSAGE_TYPES.brief,
    }


def get_submission_vacancies_qs(submission):
    if submission.source == SUBMISSION_SOURCES.form:
        return submission.form.vacancies.all()
    elif submission.source == SUBMISSION_SOURCES.publication:
        return Vacancy.unsafe.filter(id=submission.publication.vacancy_id)
    elif submission.source == SUBMISSION_SOURCES.reference:
        return submission.reference.vacancies.all()
    elif submission.source == SUBMISSION_SOURCES.rotation:
        return submission.rotation.vacancies.all()
    return Vacancy.unsafe.none()


def get_applications_waiting_contest_results(submission_ids):
    """
    По списку id откликов отдает queryset претендентств, которые ожидают результатов Контеста:
    это активные претендентства откликов с резолюцией 'Отправлено тестовое задание'
    """
    return (
        Application.unsafe
        .filter(active_applications_query)
        .filter(
            submission_id__in=submission_ids,
            resolution=APPLICATION_RESOLUTIONS.test_task_sent,
        )
    )


def filter_submissions_by_recruiter(queryset, recruiter):
    memberships_qs = VacancyMembership.unsafe.filter(
        member=recruiter,
        role__in=(
            VACANCY_ROLES.main_recruiter,
            VACANCY_ROLES.recruiter,
        ),
    )

    # Новые отклики из КФ на вакансии данного рекрутера
    is_new_form_submission = Q(
        source=SUBMISSION_SOURCES.form,
        status=SUBMISSION_STATUSES.new,
        form__in=memberships_qs.values('vacancy__submission_forms'),
    )
    is_new_publication_submission = Q(
        source=SUBMISSION_SOURCES.publication,
        status=SUBMISSION_STATUSES.new,
        publication__in=memberships_qs.values('vacancy__publications'),
    )

    # Новые отклики-рекомендации на вакансии данного рекрутера
    is_new_reference_submission = Q(
        source=SUBMISSION_SOURCES.reference,
        status=SUBMISSION_STATUSES.new,
        reference__in=memberships_qs.values('vacancy__references'),
    )
    # Новые отклики-ротации на вакансии данного рекрутера
    is_new_rotation_submission = Q(
        source=SUBMISSION_SOURCES.rotation,
        status=SUBMISSION_STATUSES.new,
        rotation__in=memberships_qs.values('vacancy__rotations'),
    )
    # Отклики, обработанные данным рекрутером
    is_closed_submission = Q(
        status=SUBMISSION_STATUSES.closed,
        responsible=recruiter,
    )
    # Отклики, где рекрутер обработал рекомендацию
    is_processed_reference_submission = Q(reference__processed_by=recruiter)

    return queryset.filter(
        is_new_form_submission
        | is_new_publication_submission
        | is_new_reference_submission
        | is_new_rotation_submission
        | is_closed_submission
        | is_processed_reference_submission
    )


def get_candidate_source_by_submission(submission):
    sources_map = {
        SUBMISSION_SOURCES.reference: SOURCES.internal_reference,
        SUBMISSION_SOURCES.rotation: SOURCES.rotation,
        SUBMISSION_SOURCES.form: SOURCES.yandex_job_website,
        SUBMISSION_SOURCES.publication: SOURCES.yandex_job_website,
    }
    return sources_map.get(submission.source, SOURCES.other)
