from intranet.femida.src.communications.choices import (
    MESSAGE_STATUSES,
    REMINDER_STATUSES,
    MESSAGE_TYPES,
)
from intranet.femida.src.communications.controllers import (
    update_or_create_internal_message,
    update_or_create_note,
    update_or_create_external_message,
    delete_external_message,
)
from intranet.femida.src.communications.models import Reminder
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.workflow import Workflow, Action


class MessageBaseAction(Action):

    def has_permission(self):
        return bool(
            self.instance.author  # исключает incoming письма
            and self.instance.author == self.user
        )


class ReminderCreateActionMixin:

    def _can_create_reminder(self):
        if hasattr(self.instance, 'user_reminders'):
            return not self.instance.user_reminders
        else:
            reminders = self.instance.reminders.filter(
                user=self.user,
                status=REMINDER_STATUSES.scheduled,
            )
            return not reminders.exists()

    def perform(self, **params):
        return Reminder.objects.create(user=self.user, **params)


###########################
# Внутренняя коммуникация #
###########################


class InternalMessageUpdateAction(MessageBaseAction):

    valid_statuses = (MESSAGE_STATUSES.sent,)

    def perform(self, **params):
        self.instance = update_or_create_internal_message(params, self.user, self.instance)
        return self.instance


class InternalMessageDeleteAction(MessageBaseAction):

    valid_statuses = (MESSAGE_STATUSES.sent,)

    def perform(self, **params):
        self.instance.status = MESSAGE_STATUSES.deleted
        self.instance.save(update_fields=['status'])
        return self.instance


class InternalMessageReminderCreateAction(ReminderCreateActionMixin, Action):

    def has_permission(self):
        return (
            self.instance.type == MESSAGE_TYPES.internal
            and self._can_create_reminder()
        )


class InternalMessageWorkflow(Workflow):

    ACTION_MAP = {
        'update': InternalMessageUpdateAction,
        'delete': InternalMessageDeleteAction,
        'reminder_create': InternalMessageReminderCreateAction,
    }


########################
# Внешняя коммуникация #
########################


class ExternalMessageUpdateAction(MessageBaseAction):

    valid_statuses = (MESSAGE_STATUSES.scheduled,)

    def perform(self, **params):
        self.instance = update_or_create_external_message(params, self.user, self.instance)
        return self.instance


class ExternalMessageDeleteAction(MessageBaseAction):

    valid_statuses = (MESSAGE_STATUSES.scheduled,)

    def perform(self, **params):
        self.instance = delete_external_message(self.instance)
        return self.instance


class ExternalMessageWorkflow(Workflow):

    ACTION_MAP = {
        'update': ExternalMessageUpdateAction,
        'delete': ExternalMessageDeleteAction,
    }


###########
# Заметки #
###########


class NoteBaseAction(MessageBaseAction):

    valid_statuses = (
        MESSAGE_STATUSES.sent,
    )

    def has_permission(self):
        is_recruiter_or_assessor = self.user.is_recruiter or self.user.is_recruiter_assessor
        return super().has_permission() and is_recruiter_or_assessor


class NoteUpdateAction(NoteBaseAction):

    def perform(self, **params):
        self.instance = update_or_create_note(params, self.user, self.instance)
        return self.instance


class NoteDeleteAction(NoteBaseAction):

    def perform(self, **params):
        self.instance.status = MESSAGE_STATUSES.deleted
        self.instance.save(update_fields=['status'])
        self.instance.reminders.filter(user=self.user).delete()
        return self.instance


class NoteReminderCreateAction(ReminderCreateActionMixin, NoteBaseAction):

    def has_permission(self):
        is_recruiter_or_assessor = self.user.is_recruiter or self.user.is_recruiter_assessor
        return is_recruiter_or_assessor and self._can_create_reminder()


class NoteWorkflow(Workflow):

    ACTION_MAP = {
        'update': NoteUpdateAction,
        'delete': NoteDeleteAction,
        'reminder_create': NoteReminderCreateAction,
    }


###############
# Напоминания #
###############


class ReminderUpdateAction(MessageBaseAction):

    def has_permission(self):
        return self.instance.user == self.user

    def perform(self, **params):
        update_instance(self.instance, params)
        return self.instance


class ReminderDeleteAction(MessageBaseAction):

    def has_permission(self):
        return self.instance.user == self.user

    def perform(self, **params):
        self.instance.delete()
        return self.instance


class ReminderWorkflow(Workflow):

    ACTION_MAP = {
        'update': ReminderUpdateAction,
        'delete': ReminderDeleteAction,
    }
