import json

from pygments import highlight
from pygments.formatters.html import HtmlFormatter
from pygments.lexers.data import JsonLexer

from django.contrib import admin
from django.utils.html import format_html
from django.utils.safestring import mark_safe

from intranet.femida.src.core import models
from intranet.femida.src.hh.admin import EntityLinkInlineAdmin


def format_url(url, name=None):
    name = name or url
    return format_html(f'<a target="_blank" href="{url}">{name}</a>')


def pretty_json(data):
    json_data = json.dumps(data, sort_keys=True, indent=2, ensure_ascii=False)
    formatter = HtmlFormatter(style='colorful')
    result = highlight(json_data, JsonLexer(), formatter)
    style = '<style>' + formatter.get_style_defs() + '</style><br>'
    return mark_safe(style + result)


def describe(description, admin_order_field=None):
    def decorator(func):
        func.short_description = description
        if admin_order_field:
            func.admin_order_field = admin_order_field
        return func
    return decorator


class CurrencyAdmin(admin.ModelAdmin):

    list_display = (
        'code',
        'name_ru',
        'name_en',
        'is_active',
    )

    list_filter = (
        'is_active',
    )

    list_editable = (
        'is_active',
    )


class CountryAdmin(admin.ModelAdmin):

    list_display = (
        'code',
        'name_ru',
        'name_en',
        'is_active',
    )

    list_filter = (
        'is_active',
    )

    list_editable = (
        'is_active',
    )


class CityAdmin(admin.ModelAdmin):

    list_display = (
        'slug',
        'name_ru',
        'name_en',
        'country',
        'country_is_active',
        'staff_id',
        'startrek_id',
        'is_active',
    )

    list_filter = (
        'is_active',
    )

    list_editable = (
        'is_active',
        'staff_id',
        'startrek_id',
    )

    inlines = (
        EntityLinkInlineAdmin,
    )

    def country_is_active(self, obj):
        return obj.country.is_active
    country_is_active.short_description = 'is country active'
    country_is_active.boolean = True


class LocationAdmin(admin.ModelAdmin):

    list_display = (
        'geo_id',
        'geocoder_uri',
        'country_code',
        'name_ru',
        'name_en',
    )

    list_filter = ('country_code',)
    search_fields = (
        'id',
        'geo_id',
        'geocoder_uri',
        'name_ru',
        'name_en',
    )


class TagAdmin(admin.ModelAdmin):

    list_display = ('name',)
    search_fields = ('name',)


class WorkModeAdmin(admin.ModelAdmin):
    list_display = ('slug', 'name_ru', 'name_en')
    search_fields = ('slug', 'name_ru', 'name_en')


admin.site.register(models.Currency, CurrencyAdmin)
admin.site.register(models.Country, CountryAdmin)
admin.site.register(models.City, CityAdmin)
admin.site.register(models.Location, LocationAdmin)
admin.site.register(models.Tag, TagAdmin)
admin.site.register(models.WorkMode, WorkModeAdmin)
