from operator import attrgetter, itemgetter

from django.db.models import AutoField
from django.db.models.fields.files import FieldFile

from intranet.femida.src.core.signals import post_bulk_create


def copy_model_instance(instance):
    """
    Копирует инстанс модели без учета m2m связей
    """
    initial = {}
    for f in instance._meta.fields:
        if isinstance(f, AutoField):
            continue
        initial[f.name] = getattr(instance, f.name)
    return instance.__class__(**initial)


def copy_model_instance_from_many(instances):
    """
    Копирует инстанс модели без учета m2m связей,
    беря данные из нескольких объектов, получая максимально полный набор данных
    """
    assert len(instances)

    initial = {}
    for f in instances[0]._meta.fields:
        if isinstance(f, AutoField):
            continue

        for instance in instances:
            value = getattr(instance, f.attname)
            if value is not None and value != '':
                initial[f.attname] = value
                break

    return instances[0].__class__(**initial)


def update_instance(instance, data, extra_update_fields=None, keep_modification_time=False):
    """
    Редактирует поля instance, перечисленные в data
    Сохраняет только те поля, которые поменялись.
    extra_update_fields сохраняются в независимости от фактического изменения полей.
    """
    # Не позволяем вызывать функцию для объектов, которых еще нет в базе (то есть нет id)
    assert instance.id is not None
    # Не надо пытаться отредактировать id у сущности
    data.pop('id', None)

    update_fields = set(extra_update_fields or [])

    for attr_name, attr_value in data.items():
        field = getattr(instance, attr_name)
        if isinstance(field, FieldFile):
            field.name = attr_value
            update_fields.add(attr_name)
        else:
            old_value = getattr(instance, attr_name)
            if old_value != attr_value:
                setattr(instance, attr_name, attr_value)
                update_fields.add(attr_name)

    # TODO: Пока не решен тикет FEMIDA-2450, нужно явно апдейтить modified,
    # если есть хотя бы еще одно поле для апдейта
    modified_field = getattr(instance, 'modified', None)
    if not keep_modification_time and modified_field and update_fields:
        update_fields.add('modified')

    # Если update_fields окажется пустым, то записи в базу не будет
    instance.save(update_fields=list(update_fields))
    return instance


def update_list_of_instances(model, queryset, data, delete_missing=True,
                             identifier=None, delete_func=None):
    """
    Редактирует список сущностей, накладывая изменения на queryset.
    Если в data есть сущность, соответствующая identifier,
    она будет отредактирована.
    Если такой нет, то она будет создана.
    Если сущность была в базе, но ее не оказалось в data, то смотрим на флаг delete_missing.
    Если он True, то удаляем эти сущности с помощью функции delete_func, иначе оставляем.
    """
    if identifier is None:
        identifier = ('id',)

    if delete_func is None:
        delete_func = lambda queryset: queryset.delete()

    get_key_by_attr = attrgetter(*identifier)
    get_key_by_item = itemgetter(*identifier)

    entities_map = {}  # маппинг идентификатор: инстанс для queryset
    missing_entities_ids = set()  # сущности из базы, которым не нашлось соответствия
    handled_keys = set()  # обработанные идентификаторы, чтобы апдейтить только первый
    for entity in queryset:
        entities_map[get_key_by_attr(entity)] = entity
        missing_entities_ids.add(entity.id)

    new_instances = []

    for entity_data in data:
        try:
            key = get_key_by_item(entity_data)
        except KeyError:
            # Если не получилось достать идентификатор, то будем создавать сущность.
            # В случае с id это ок, а вот в случае сложных идентификаторов - вопрос.
            instance = None
        else:
            instance = entities_map.get(key)

        if instance is not None:
            if key not in handled_keys:
                update_instance(instance, entity_data)
                missing_entities_ids.discard(instance.id)
                handled_keys.add(key)
        else:
            entity_data.pop('id', None)
            new_instances.append(model(**entity_data))

    if delete_missing and missing_entities_ids:
        delete_func(queryset.filter(id__in=missing_entities_ids))

    created_objs = model.objects.bulk_create(new_instances)
    post_bulk_create.send(
        sender=model,
        queryset=created_objs,
    )
