from django.db.models import Subquery, OuterRef, Count, IntegerField
from django.db.models.functions import Coalesce


def get_count_subquery(queryset, reverse_related_name):
    """
    Позволяет аннотировать Subquery с COUNT

    qs = (
        Candidates.unsafe
        .annotate(
            cons_count=get_count_subquery(
                queryset=Consideration.unsafe.all(),
                reverse_related_name='candidate_id',
            )
        )
        .values('id', 'cons_count')
    )

    транслируется в такой SQL:

    SELECT
        c.id,
        COALESCE(
            (
                SELECT COUNT(t.id) AS count
                FROM candidates_consideration as t
                WHERE t.candidate_id = c.id
                GROUP BY t.candidate_id
            ),
            0
        ) AS cons_count
    FROM candidates_candidate as c;

    Подробнее можно почитать здесь:
    https://stackoverflow.com/questions/42543978/django-1-11-annotating-a-subquery-aggregate
    """
    subquery = (
        queryset
        .filter(**{reverse_related_name: OuterRef('id')})
        .order_by()
        .values(reverse_related_name)
        .annotate(count=Count('id'))
        .values('count')
    )
    return Coalesce(
        Subquery(
            subquery,
            output_field=IntegerField(),
        ),
        0
    )
