import time
import ylock
import logging

import importlib
from celery.app.task import Task

from django.core.management.base import BaseCommand, CommandError
from django.conf import settings


logger = logging.getLogger(__name__)
manager = ylock.backends.create_manager(**settings.YLOCK)


class Command(BaseCommand):

    help = 'Run celery task by name.'

    def add_arguments(self, parser):
        parser.add_argument('--task')
        parser.add_argument('-s', '--synchronous', action='store_true')
        parser.add_argument('--delay', default=10, type=int)
        parser.add_argument('--timeout', default=10, type=int)
        parser.add_argument('--block', action='store_true')

    def run_task(self, task, is_synchronous, delay):
        if is_synchronous:
            task()
            print('Task was executed synchronously')
        else:
            print('Putting task in a queue with delay...', end=' ')
            task.delay()
            # Чтобы lock не освобождался слишком быстро
            time.sleep(delay)
            print('OK')

    def run_task_with_lock(self, task, options):
        task_name = options.get('task')
        lock_name = f'{settings.DEPLOY_STAGE_ID}.{task_name}'
        is_synchronous = options.get('synchronous')
        delay = options.get('delay')
        timeout = options.get('timeout')
        is_blocking = options.get('block')

        logger.info('Acquiring lock `%s`', lock_name)
        with manager.lock(lock_name, timeout, is_blocking) as acquired:
            if acquired:
                logger.info('Lock `%s` acquired', lock_name)
                self.run_task(task, is_synchronous, delay)
            else:
                logger.info('Lock `%s` NOT acquired', lock_name)

    def handle(self, *args, **options):
        task_name = options.get('task')

        if not task_name:
            raise CommandError('--task argument must be specified')

        module, task = task_name.rsplit('.', 1)
        module = importlib.import_module(module)
        task = getattr(module, task)

        if not isinstance(task, Task):
            raise CommandError('This is not a celery task.')

        self.run_task_with_lock(task, options)
