from django.db import models
from model_utils.models import TimeStampedModel

from intranet.femida.src.core.managers import ActiveManager
from intranet.femida.src.utils.translation import get_language


class I18NModelMixin:
    """
    Миксин для добавления автоматической локализации.

    Для моделей с локализованными полями формата field_ru, field_en и т.д.
    добавляется доп.атрибут формата field,
    который берет данные из поля в текущей локали.
    По умолчанию, если поля в нужной локали нет, используется локаль "ru".

    Для использования, в подклассах нужно
    явно указать для каких полей это применимо.
    Напр.:
    localized_fields = ('field1', 'field2')
    """

    localized_fields = []

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        for item in self.localized_fields:
            default_lang = 'ru'
            lang = get_language(default_lang)
            field_name = '{field}_{lang}'.format(field=item, lang=lang)
            if not hasattr(self, field_name):
                field_name = '{field}_{lang}'.format(field=item, lang=default_lang)
            if hasattr(self, item):
                item = f'localized_{item}'
                assert not hasattr(self, item)

            setattr(self, item, getattr(self, field_name))


class I18NNameModelMixin(I18NModelMixin):

    localized_fields = ('name',)


class I18NLocalizedNameModelMixin(I18NNameModelMixin):

    @property
    def name_ru(self):
        return self.name


class Currency(I18NNameModelMixin, models.Model):

    objects = models.Manager()
    active = ActiveManager()

    code = models.CharField(max_length=3)
    name_ru = models.CharField(max_length=64)
    name_en = models.CharField(max_length=64)
    startrek_id = models.CharField(max_length=2, null=True, blank=True)
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return '%s (%s)' % (self.code, self.name_ru)


class Country(I18NNameModelMixin, models.Model):

    objects = models.Manager()
    active = ActiveManager()

    code = models.CharField(max_length=2, unique=True)
    name_ru = models.CharField(max_length=100)
    name_en = models.CharField(max_length=100)
    is_active = models.BooleanField(default=True)

    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        if self.is_active is False:
            self.cities.update(is_active=False)

    def __str__(self):
        return self.name_ru


class City(I18NNameModelMixin, models.Model):

    objects = models.Manager()
    active = ActiveManager()

    country = models.ForeignKey(
        to=Country,
        on_delete=models.CASCADE,
        related_name='cities',
    )
    name_ru = models.CharField(max_length=100)
    name_en = models.CharField(max_length=100)

    staff_id = models.IntegerField(null=True, blank=True)
    startrek_id = models.IntegerField(null=True, blank=True)
    is_active = models.BooleanField(default=True)

    slug = models.CharField(max_length=255, unique=True, null=True)

    def __str__(self):
        return self.name_ru


class Tag(TimeStampedModel):

    name = models.CharField(max_length=255, unique=True)

    def __str__(self):
        return self.name

    class Meta:
        ordering = ('name',)


class Location(I18NNameModelMixin, models.Model):

    geo_id = models.IntegerField(unique=True)
    geocoder_uri = models.CharField(max_length=255)
    country_code = models.CharField(max_length=2)

    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)

    def __str__(self):
        return self.name


class WorkMode(I18NNameModelMixin, models.Model):

    name_ru = models.CharField(max_length=32)
    name_en = models.CharField(max_length=32)

    slug = models.CharField(max_length=32, unique=True, null=False)

    def __str__(self):
        return self.name


class LanguageTag(I18NNameModelMixin, TimeStampedModel):

    """
    Тэг языка по IETF BCP 47 Language tag.

    Пока поддерживаем только часть стандарта:
    двухбуквенный сабтэг языка по ISO639-1
    остальные сабтеги пока игнорируем

    https://www.iana.org/assignments/language-subtag-registry/language-subtag-registry
    https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes
    """
    tag = models.CharField(max_length=2, unique=True)
    native = models.CharField(max_length=64)
    name_en = models.CharField(max_length=64)
    name_ru = models.CharField(max_length=64)
