from intranet.femida.src.notifications.utils import get_base_context
from .exceptions import FemidaError


class WorkflowError(FemidaError):
    """
    Нарушение воркфлоу модели
    """


class ActionProhibitedError(WorkflowError):
    """
    Экшен не прошел проверку is_available
    """


class Action:
    """
    Класс, описывающий действие над неким объектом instance, выполняемое
    пользователем user.
    is_visible - если True, то user может попыться совершить действие.
    is_available - если True, то user на самом деле может совершить действие.
    В общем случае is_visible и is_available совпадают. Но иногда необходимо показать
    пользователю кнопку, даже если действие он совершить не может.

    perform - совершает его, независимо от ответа is_available.
    """
    valid_statuses = None
    status_field_name = 'status'

    def __init__(self, wf, **kwargs):
        self.wf = wf
        self.instance = self.wf.instance
        self.user = self.wf.user
        self.extra_data = {}

    def is_status_correct(self):
        """
        Проверяем доступы, связанные непостредственно со статусами
        """
        if self.valid_statuses is None:
            return True
        return getattr(self.instance, self.status_field_name, None) in self.valid_statuses

    def has_permission(self):
        """
        Проверяем доступы, связанные с правами пользователя или другими полями объекта
        """
        return True

    def is_visible(self):
        return self.is_status_correct() and self.has_permission()

    def is_available(self):
        return self.is_visible()

    def perform(self, **params):
        return

    def get_context(self):
        base_context = get_base_context()
        base_context.update(instance=self.instance, user=self.user)
        return base_context

    def raise_error(self, code='', error_class=None, extra_context=None, fail_silently=False):
        extra_context = extra_context or {}
        error_class = error_class or WorkflowError

        if fail_silently:
            if 'warnings' not in self.extra_data:
                self.extra_data['warnings'] = []
            self.extra_data['warnings'].append({
                'code': code,
                'params': extra_context,
            })
        else:
            raise error_class(code)

    __call__ = perform


class Workflow:
    """
    Точка входа для пула экшенов какой-то сущности.
    Содержит в себе глобальные данные, необходимые для каждого экшена.
    """
    ACTION_MAP = None

    def __init__(self, instance, user, **kwargs):
        self.instance = instance
        self.user = user
        # В конcтруктор Workflow надо прокидывать данные, актуальные для любого экшена
        self.kwargs = kwargs

    def get_action(self, action_name):
        if action_name not in self.ACTION_MAP:
            raise WorkflowError('Invalid action name.')
        action_class = self.ACTION_MAP[action_name]
        return action_class(wf=self)

    def perform_action(self, action_name, strict=True, **params):
        action = self.get_action(action_name)
        if strict:
            is_available = action.is_available()
            if not is_available:
                raise ActionProhibitedError('action_prohibited')
        return action.perform(**params)

    def perform_action_if_possible(self, action_name, **params):
        """
        Запускаем экшн, если прошли проверку is_available, иначе просто игнорируем.
        """
        try:
            return self.perform_action(action_name, **params)
        except ActionProhibitedError:
            return

    def get_actions_visibility(self):
        result = {}
        for action_name, action_class in self.ACTION_MAP.items():
            result[action_name] = action_class(wf=self).is_visible()
        return result
