import logging
import os

from typing import Dict, Optional
from urllib.parse import urlparse, parse_qs

import requests
import waffle
from dateutil.relativedelta import relativedelta
from django.conf import settings
from django.core.exceptions import ValidationError
from django.core.files.base import ContentFile
from django.db.models import ObjectDoesNotExist
from django.utils import timezone
from requests import RequestException

from intranet.femida.src.attachments.models import Attachment
from intranet.femida.src.candidates.choices import (
    SUBMISSION_SOURCES,
    VERIFICATION_STATUSES,
    VERIFICATION_RESOLUTIONS,
    VERIFICATION_TYPES,
)
from intranet.femida.src.candidates.models import (
    Candidate,
    CandidateSubmission,
    DuplicationCase,
    Verification,
)
from intranet.femida.src.candidates.signals import verification_on_check, verification_succeeded
from intranet.femida.src.core.switches import TemporarySwitch
from intranet.femida.src.candidates.tasks import (
    auto_handle_internship_submission,
    reupload_file_to_mds_task,
    create_ess_issue_task,
)
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.monitoring.utils import alarm
from intranet.femida.src.notifications.candidates import (
    send_verification_data_to_vendor,
    VerificationMaybeDuplicateNotification,
    VerificationSubmittedNotification,
    VerificationSuccessNotification,
    VerificationWarningNotification,
)
from intranet.femida.src.oebs.choices import LOGIN_LOOKUP_DOCUMENT_TYPES
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.startrek.operations import IssueTransitionOperation, IssueUpdateOperation
from intranet.femida.src.startrek.utils import TransitionEnum, ResolutionEnum
from intranet.femida.src.utils.datetime import shifted_now
from intranet.femida.src.vacancies.models import SubmissionForm


if os.getenv('IS_ARCADIA'):
    from django_mds.client import APIError
else:
    from mds import APIError


logger = logging.getLogger(__name__)


def _get_verification_by_uuid(verification_uuid):
    try:
        return (
            Verification.objects
            .filter(uuid=verification_uuid)
            .select_related('candidate')
            .first()
        )
    except ValidationError:
        return None


def _get_instance_by_id_or_raise(model_class, instance_id):
    try:
        return model_class.objects.get(id=instance_id)
    except ObjectDoesNotExist:
        logger.error('%s for id %s does not exist', model_class, instance_id)
        raise


def handle_submission_form_answer(data):
    """
    Пример data:

    {
      "params": {
        "form_id": "670",
        "cand_surname": "Кипарис",
        "cand_phone": "+7 905 790-00-91",
        "cand_cv": "https://study.yandex-team.ru/files?path=%2F11913%2F08de1db_shikardyatina.png",
        "cand_info": "про себя пара слов",
        "form_url": "https://forms.yandex-team.ru/surveys/670/?iframe=1",
        "publication_url": https://yandex.ru/jobs/vacancies/proj_man/pm_optimization/",
        "cand_email": "kiparis@yandex-team.ru",
        "cand_questions": "Это вопрос номер раз:\nответ раз\n\nА это второй вопрос:\nответ два",
        "cand_name": "Андрей"
      },
      "jsonrpc": "2.0",
      "method": "POST",
      "id": "583819466162d74119f22ec2"
    }
    """
    form_data = data['params']
    attachment_url = form_data.get('cand_cv')
    is_reupload_failed = False
    publication = None
    submission_form = None

    if attachment_url:
        try:
            attachment = reupload_file_to_mds(attachment_url)
        except (RequestException, APIError):
            logger.exception('Failed to reupload file to mds. Trying to do it asynchronously.')
            attachment = None
            is_reupload_failed = True
    else:
        attachment = None

    if form_data.get('publication_id'):
        publication = _get_instance_by_id_or_raise(Publication, form_data['publication_id'])
    else:
        submission_form = _get_instance_by_id_or_raise(SubmissionForm, form_data['form_id'])

    if CandidateSubmission.unsafe.filter(forms_answer_id=data['id']).exists():
        logger.warning('Submission with current Forms answer ID (%s) already exists', data['id'])
        return

    submission = CandidateSubmission.objects.create(
        form=submission_form,
        publication=publication,
        attachment=attachment,
        forms_data=data,
        forms_answer_id=data['id'],
        publication_url=form_data.get('publication_url', ''),
        first_name=form_data.get('cand_name', ''),
        last_name=form_data.get('cand_surname', ''),
        phone=form_data.get('cand_phone', ''),
        email=form_data.get('cand_email', ''),
        comment=form_data.get('cand_info', ''),
        source=SUBMISSION_SOURCES.publication if publication else SUBMISSION_SOURCES.form,
    )
    if submission.is_internship:
        auto_handle_internship_submission.delay(submission_id=submission.id)

    if is_reupload_failed:
        reupload_file_to_mds_task.delay(submission.id)


# TODO: на рефакторинг
def handle_verification_form_answer(data):
    """
    Обработка ответа из отправленной кандидату формы КИ
    """
    now = timezone.now()
    month_later = now + relativedelta(days=30)

    form_data = data['params']
    verification_uuid = form_data.get('uuid')
    verification = _get_verification_by_uuid(verification_uuid)

    if not verification:
        logger.warning(
            'Unknown verification_uuid was received from Forms Constructor: %s',
            verification_uuid,
        )
        return
    error_message = _check_verification(verification, now)
    if error_message is not None:
        notification = VerificationWarningNotification(
            instance=verification,
            comment=error_message,
        )
        notification.send()
        return

    for document_type, _ in LOGIN_LOOKUP_DOCUMENT_TYPES:
        form_data[document_type] = form_data.get(document_type, '').strip()

    skip_vendor_check = settings \
        .VERIFICATION_SETTINGS[verification.type or VERIFICATION_TYPES.default]['workflow']\
        .get('skip_vendor_check', False)

    # Skips on_check status to skip vendor check phase
    new_status = VERIFICATION_STATUSES.on_check
    if skip_vendor_check:
        new_status = VERIFICATION_STATUSES.on_ess_check

    update_data = dict(
        raw_data=data,
        status=new_status,
        # Продлеваем срок действия ссылки
        link_expiration_date=month_later,
        # Фиксируем время и дату отправки на проверку вендору
        sent_on_check=now,
    )
    update_instance(verification, update_data)

    # Создаем ESS тикет, отправляем данные компании-вендору и уведомление рекрутеру
    create_ess_issue_task.delay(verification.id)

    if not skip_vendor_check:
        send_verification_data_to_vendor(verification)

    VerificationSubmittedNotification(verification).send()
    verification_on_check.send(
        sender=Verification,
        verification=verification,
        candidate=verification.candidate,
    )


def _check_verification(verification, frozen_now):
    if verification.status != VERIFICATION_STATUSES.new:
        logger.info(
            'Forms Constructor request for already filled in verification. '
            'verification_uuid: %s, Verification: %s',
            verification.uuid, verification.id,
        )
        return 'анкета на КИ уже заполнялась кандидатом'
    if frozen_now > verification.link_expiration_date:
        logger.info(
            'Forms Constructor request for verification with expired link. '
            'verification_uuid: %s, Verification: %s',
            verification.uuid, verification.id,
        )
        return 'срок действия анкеты на КИ истёк'


class VerificationHandler:
    """
    Обработка ответа вендора после проверки КИ
    """
    OK_TAGS = [settings.STARTREK_ESS_VENDOR_OK_TAG]
    NOT_OK_TAGS = [settings.STARTREK_ESS_VENDOR_NOT_OK_TAG, settings.STARTREK_ESS_YELLOW_TAG]

    def __init__(self, verification: Verification, result: Dict):
        self._verification = verification
        self._result = result

    def mark_finished(self, ok: bool) -> None:
        self._verification.status = VERIFICATION_STATUSES.on_ess_check
        self._verification.save(update_fields=['status', 'modified'])
        self._update_ess_ticket(ok)

    def _update_ess_ticket(self, ok: bool) -> None:
        operation = IssueUpdateOperation(self._verification.startrek_ess_key)
        operation.delay(
            comment=self._result,
            tags={
                'add': self.OK_TAGS if ok else self.NOT_OK_TAGS,
            },
        )


# TODO: Убрать после релиза FEMIDA-7226
#       Не забыть убрать settings.STARTREK_ESS_GREEN_TAG
#       из 142-startrek.conf
def _handle_verification_ok_old(verification, result):
    """
    Обработка положительного ответа вендора после проверки КИ
    """
    update_data = dict(
        status=VERIFICATION_STATUSES.closed,
        resolution=VERIFICATION_RESOLUTIONS.hire,
        expiration_date=shifted_now(months=3),
    )
    update_instance(verification, update_data)

    operation = IssueTransitionOperation(verification.startrek_ess_key)
    operation.delay(
        transition=TransitionEnum.close,
        resolution=ResolutionEnum.fixed,
        comment=result,
        tags={
            'add': [settings.STARTREK_ESS_GREEN_TAG],
        },
    )
    VerificationSuccessNotification(verification).send()
    verification_succeeded.send(
        sender=Verification,
        verification=verification,
        candidate=verification.candidate,
    )


# TODO: убрать после релиза FEMIDA-7226
def _handle_verification_not_ok_old(verification, result):
    """
    Обработка отрицательного ответа вендора после проверки КИ.
    Переводим ESS-тикет на ручной анализ.
    """
    verification.status = VERIFICATION_STATUSES.on_ess_check
    verification.save(update_fields=['status', 'modified'])

    operation = IssueTransitionOperation(verification.startrek_ess_key)
    operation.delay(
        transition=TransitionEnum.need_info,
        comment=result,
        tags={
            'add': [settings.STARTREK_ESS_YELLOW_TAG],
        },
    )


class VerificationResult:
    """
    Обработка ответа в форме результата проверки КИ, которую заполняет вендор
    :param data: данные из Форм
    :param ok: результат проверки
    """
    def __init__(self, data, ok):
        params = data['params']
        self._result = params.get('result')
        self._verification_uuid = params.get('uuid')
        self._ok = ok
        self._verification = None

    def update_verification(self) -> None:
        if not self._get_verification_by_uuid():
            self._alarm_unknown_uuid()
            return

        self._check_inn_duplicate()
        self._save_candidate_inn()

        if self._verification_status_is_wrong():
            self._notify('вендор вынес повторный вердикт')
            return

        if self._verification_is_expired():
            self._notify('вендор вынес вердикт по кандидату, но срок проверки истёк')
            return

        # TODO: убрать свитч после релиза FEMIDA-7226
        if waffle.switch_is_active(TemporarySwitch.NEW_VERIFICATION_FLOW):
            handler = VerificationHandler(self._verification, self._result)
            handler.mark_finished(self._ok)
        else:
            if self._ok:
                _handle_verification_ok_old(self._verification, self._result)
            else:
                _handle_verification_not_ok_old(self._verification, self._result)

    def _get_verification_by_uuid(self) -> Optional[Verification]:
        self._verification = _get_verification_by_uuid(self._verification_uuid)
        return self._verification

    def _alarm_unknown_uuid(self) -> None:
        alarm('Vendor sent {result} for verification with unknown uuid: {uuid}'.format(
            result='OK' if self._ok else 'NOT OK',
            uuid=self._verification_uuid,
        ))

    def _save_candidate_inn(self) -> None:
        candidate = self._verification.candidate
        candidate.inn = self._verification.inn
        candidate.save()

    def _check_inn_duplicate(self) -> None:
        inn = self._verification.inn
        candidate = self._verification.candidate
        potential_former_employee = (
            Candidate.unsafe.alive()
                            .filter(inn=inn)
                            .exclude(id=candidate.id)
                            .first()
        )
        if potential_former_employee:
            DuplicationCase.objects.create(
                first_candidate=candidate,
                second_candidate=potential_former_employee,
                is_active=True,
                is_auto_merge=False,
            )
            self._notify_inn_duplicate()

    def _notify_inn_duplicate(self) -> None:
        VerificationMaybeDuplicateNotification(
            self._verification
        ).send()

    def _verification_status_is_wrong(self) -> bool:
        # Note: в теории, такое может произойти наоборот при преждевременном
        # заполнении формы. Но на практике такое практически невозможно
        # – вендору неоткуда взять uuid
        return self._verification.status != VERIFICATION_STATUSES.on_check

    def _verification_is_expired(self) -> bool:
        return timezone.now() > self._verification.link_expiration_date

    def _notify(self, comment: str) -> None:
        notification = VerificationWarningNotification(
            instance=self._verification,
            comment=comment,
        )
        notification.send()


def reupload_file_to_mds(url):
    """
    Переливает файл из MDS конструктора форм в наш MDS
    """
    headers = {'Authorization': 'OAuth {}'.format(settings.FEMIDA_ROBOT_TOKEN)}
    query = urlparse(url).query
    url = '{base_url}?{query}'.format(base_url=settings.FORMS_API_URL, query=query)

    try:
        response = requests.get(
            url=url,
            headers=headers,
            verify=settings.YANDEX_INTERNAL_CERT_PATH,
            timeout=15,
        )
        response.raise_for_status()
    except RequestException:
        logger.exception('Failed to download file from %s', url)
        raise

    default_name = 'resume'
    attached_file = ContentFile(response.content, name=default_name)
    parsed_query = parse_qs(query)

    try:
        # Достаем из url параметр path и вытаскиваем из него название без хэша
        file_name = parsed_query['path'][0].split('_', 1)[1]
    except (KeyError, IndexError):
        logger.exception('Failed trying parse filename from %s', url)
        file_name = default_name

    return Attachment.objects.create(
        name=file_name,
        size=attached_file.size,
        attached_file=attached_file,
    )
