import waffle

from intranet.femida.src.problems.helpers import has_access_to_problems


class BaseNavBlock:
    """
    Базовый класс для блоков в навигации
    """
    name = None
    url = None
    alias = None
    external = False
    include_self = True

    def __init__(self, *subs):
        self.subs = subs

    def is_available(self, user):
        return True

    def _get_dict(self, alias=None):
        result = {
            'url': self.url,
            'name': self.name,
        }
        if alias:
            result['alias'] = alias
        if self.external:
            result['external'] = self.external
        return result

    def build(self, user):
        if not self.is_available(user):
            return {}
        subs = (s.build(user) for s in self.subs)
        subs = [s for s in subs if s]
        if subs:
            if self.include_self:
                subs = [self._get_dict(self.alias)] + subs
            return {
                'name': self.name,
                'subs': subs,
            }
        return self._get_dict()


class SimpleBlock(BaseNavBlock):

    def __init__(self, name, url=None, alias=None, *subs):
        super().__init__(*subs)
        self.name = name
        self.url = url or f'/{name}/'
        self.alias = alias


class Menu(BaseNavBlock):

    include_self = False

    def build(self, user):
        return super().build(user).get('subs', [])


class CandidateBlock(BaseNavBlock):

    name = 'candidates'
    url = '/candidates/'
    alias = 'search'


class CandidateDashboardBlock(BaseNavBlock):

    name = 'candidates-dashboard'
    url = '/candidates/dashboard/'

    def is_available(self, user):
        return user.is_recruiter or user.is_recruiter_assessor


class CandidateFilterBlock(BaseNavBlock):

    name = 'candidates-filter'
    url = '/candidates/filter/'

    def is_available(self, user):
        return waffle.switch_is_active('enable_candidates_filter')


class CandidateCreateBlock(BaseNavBlock):

    name = 'candidate-create'
    url = '/candidates/create/'

    def is_available(self, user):
        return user.is_recruiter or user.is_recruiter_assessor


class DuplicationCaseBlock(BaseNavBlock):

    name = 'dup-cases'
    url = '/candidates/duplicates/'

    def is_available(self, user):
        if waffle.switch_is_active('is_rkn'):
            return False
        return user.is_recruiter or user.is_recruiter_assessor


class SubmissionBlock(BaseNavBlock):

    name = 'submissions'
    url = '/submissions/'

    def is_available(self, user):
        return user.is_recruiter or user.is_recruiter_assessor


class VacancyBlock(BaseNavBlock):

    name = 'vacancies'
    url = '/vacancies/'
    alias = 'my-vacancies'


class PublicationBlock(BaseNavBlock):

    name = 'publications'
    url = '/vacancies/publications/'

    def is_available(self, user):
        if not waffle.switch_is_active('enable_publication_restriction'):
            return True
        return user.has_perm('permissions.can_see_publications')


class ProblemBlock(BaseNavBlock):

    name = 'problems'
    url = '/problems/'
    alias = 'catalog'

    def is_available(self, user):
        return has_access_to_problems(user)


class StatsBlock(BaseNavBlock):

    name = 'statistics'
    url = 'https://stat.yandex-team.ru/Dashboard/Femida'
    external = True
