import json
import logging

from django.conf import settings
from django.contrib.auth.mixins import PermissionRequiredMixin
from django.http import Http404
from django.http.response import HttpResponseRedirect, HttpResponseBadRequest
from django.utils.decorators import method_decorator
from django.utils.functional import cached_property
from django.views.generic.base import TemplateView
from django.views.decorators.csrf import ensure_csrf_cookie
from django.shortcuts import redirect

from constance import config
import waffle
from intranet.femida.src.utils.cache import memoize

from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.hh.mixins import HHAuthenticateMixin
from intranet.femida.src.notifications.utils import get_base_context
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.startrek.helpers import get_short_issue_changelog
from intranet.femida.src.startrek.utils import get_issue, StartrekError
from intranet.femida.src.utils.authdata import AuthDataMixin
from intranet.femida.src.utils.csp import CSPMixin
from intranet.femida.src.utils.frontend import get_manifest


logger = logging.getLogger(__name__)


@memoize(5 * 60)
def _get_manifest(pr_number):
    return get_manifest(pr_number)


@method_decorator(ensure_csrf_cookie, name='dispatch')
class IndexViewBase(CSPMixin, TemplateView):

    template_name = None
    bundle = None
    bundle_rum_code = None

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'vendor_packages': json.loads(config.VENDOR_PACKAGES),
            'bundle': self.bundle,
            'bundle_rum_code': self.bundle_rum_code,
            'nonce': self.csp_nonce,
            'reqid': self.reqid,
        })
        self.set_static_url(context)
        context['auth_data'] = self.get_auth_data()
        return context

    def set_static_url(self, context):
        """
        Если хост начинается с femida-front-pr, то считаем, что это стенд.
        В этом случае отправляем за статикой yastat, но по специальному пути,
        сформированному из номера pr и хеша коммита.
        """
        yastat_url = f'//yastat.net/{config.FRONT_STATIC_PATH}'

        host = self.request.META['HTTP_HOST']
        is_stand = host.startswith('femida-front-pr')

        if is_stand:
            domain_part = host.split('.')[0]
            parts = domain_part.replace('femida-front-pr-', '', 1).split('-')
            pr_number = parts[0]
            context['front_static_assets'] = _get_manifest(pr_number)
        else:
            context['front_static_assets'] = json.loads(config.FRONT_STATIC_ASSETS)
        context['static_url_prefix'] = yastat_url

    def get(self, request, *args, **kwargs):
        if (self.bundle == 'candidates'
                and request.path.startswith('/candidates/duplicates')
                and waffle.switch_is_active('is_rkn')):
            raise Http404
        response = super().get(request, *args, **kwargs)
        self.set_csp_header(response)
        return response


class IndexView(AuthDataMixin, IndexViewBase):

    template_name = 'index.html'
    bundle = 'femida'
    bundle_rum_code = 3570

    csp_connect_sources = [
        "'self'",
        'yandex.ru',
        'mc.yandex.ru',
        'mc.webvisor.com',
        'mc.webvisor.org',
        'orange.yandex-team.ru',
        'search.yandex-team.ru',
        'staff.yandex-team.ru',
        'staff-api.yandex-team.ru',
        'cloud-api.yandex-team.ru',
        'center.yandex-team.ru',
        'q.yandex-team.ru',
        'api.stat.yandex-team.ru',
        'backend.messenger.yandex-team.ru',
        settings.WF_HOST,
        settings.JING_HOST,
    ]
    csp_font_sources = [
        'yastatic.net',
        'yastat.net',
    ]
    csp_img_sources = [
        "'self'",
        'data:',
        'yastatic.net',
        'yastat.net',
        'http://img.yandex.net',
        'mc.yandex.ru',
        'mc.webvisor.com',
        'mc.webvisor.org',
        '*.yandex-team.ru',
    ]
    csp_script_sources = [
        "'unsafe-eval'",
        'yastatic.net',
        'yastat.net',
        'mc.yandex.ru',
        'mc.webvisor.com',
        'mc.webvisor.org',
        'betastatic.yastatic.net',
        'center.yandex-team.ru',
        'staff.yandex-team.ru',
        'static.yandex.net',
        'gcn.yandex-team.ru',
        'search.yandex-team.ru',
        'chat.s3.yandex.net',
        'wfaas.yandex-team.ru',
    ]
    csp_style_sources = [
        "'unsafe-inline'",
        'yastatic.net',
        'yastat.net',
    ]
    csp_child_sources = [
        'blob:',
        'mc.yandex.ru',
    ]
    csp_frame_sources = [
        'blob:',
        'yastat.net',
        'mc.yandex.ru',
        '*.yandex-team.ru',
    ]


class CandidateDetailIndexView(IndexView):
    """
    Редиректит на оригинал кандидата, если url указывает на смерженного
    https://st.yandex-team.ru/FEMIDA-4177
    """
    bundle = 'candidates'
    bundle_rum_code = 3571

    def get(self, request, *args, **kwargs):
        redirection = self._get_redirect()
        if redirection is not None:
            return redirection
        else:
            return super().get(request, *args, **kwargs)

    def _get_redirect(self):
        if 'no_redirect' in self.request.GET:
            return None
        original_id = (
            Candidate.unsafe
            .filter(
                id=self.kwargs['candidate_id'],
                is_duplicate=True,
            )
            .values_list('original_id', flat=True)
            .first()
        )
        if original_id is None:
            return None
        response = redirect('candidate-detail', candidate_id=original_id)
        response['Location'] += '/?redirected_from={duplicate}'.format(
            duplicate=self.kwargs['candidate_id'],
        )
        return response


class CandidateCreateIndexView(HHAuthenticateMixin, IndexView):

    bundle = 'candidates'
    bundle_rum_code = 3571

    def get(self, request, *args, **kwargs):
        if 'hh_id' in request.GET:
            is_hh_authenticated, redirect_url = self.get_hh_authentication_details()
            if not is_hh_authenticated:
                return HttpResponseRedirect(redirect_url)
        return super().get(request, *args, **kwargs)


@method_decorator(ensure_csrf_cookie, name='dispatch')
class ActionLogIndexView(TemplateView):

    template_name = 'actionlog.html'


class PublicationOGImageView(IndexView):

    bundle = 'publications'
    bundle_rum_code = 4067

    @cached_property
    def publication(self):
        return get_object_or_40x(Publication, uuid=self.kwargs['uuid'])

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['LANGUAGE_CODE'] = self.publication.lang
        return context


class StChangelogIndexView(PermissionRequiredMixin, TemplateView):
    """
    Страница просмотра истории тикета из Трекера
    """
    permission_required = 'permissions.can_view_startrek_changelog'
    template_name = 'st-changelog.html'

    def get_context_data(self, **kwargs):
        context = get_base_context()
        issue = get_issue(kwargs['key'].upper())
        context['issue'] = issue
        context['changelog'] = get_short_issue_changelog(issue)
        context['transitions'] = list(issue.transitions.get_all())
        return context

    def get(self, request, *args, **kwargs):
        try:
            return super().get(request, *args, **kwargs)
        except StartrekError as exc:
            return HttpResponseBadRequest(exc.message.encode('utf-8'))
