import logging
import requests

from datetime import timedelta
from base64 import b64encode
from urllib.parse import urlencode

from django.conf import settings
from django.contrib.sites.models import Site
from django.urls.base import reverse
from django.utils import timezone

from intranet.femida.src.hh.exceptions import HeadhunterOauthError
from intranet.femida.src.hh.models import HHOauthToken


logger = logging.getLogger(__name__)


class HeadhunterOAuth:

    @classmethod
    def _get_redirect_uri(cls):
        return '{protocol}://{host}{url}'.format(
            protocol=settings.FEMIDA_PROTOCOL,
            host=Site.objects.get_current().domain,
            url=reverse('api:hh:authorize'),
        )

    @classmethod
    def _post(cls, method, data):
        url = settings.HH_OAUTH_URL + method
        headers = {
            'User-Agent': settings.HH_USER_AGENT,
        }
        data.update({
            'client_id': settings.HH_CLIENT_ID,
            'client_secret': settings.HH_CLIENT_SECRET,
        })

        try:
            response = requests.post(
                url=url,
                data=data,
                headers=headers,
                timeout=10,
            )
        except (requests.ConnectionError, requests.Timeout):
            logger.exception('HH oauth is not responding')
            raise HeadhunterOauthError('hh_oauth_no_response')

        try:
            response_data = response.json()
        except ValueError:
            logger.exception('Error during parsing HH oauth response')
            raise HeadhunterOauthError('hh_oauth_invalid_response')

        if 'error' in response_data:
            raise HeadhunterOauthError('hh_oauth_%s' % response_data['error'])

        return response_data

    @classmethod
    def get_oauth_url(cls, return_path=None):
        params = {
            'response_type': 'code',
            'client_id': settings.HH_CLIENT_ID,
            'redirect_uri': cls._get_redirect_uri(),
        }
        if return_path:
            params['state'] = b64encode(return_path)
        return settings.HH_OAUTH_URL + 'authorize?' + urlencode(params)

    @classmethod
    def get_token_refresh_url(cls, return_path=None):
        params = {}
        if return_path:
            params['state'] = b64encode(return_path)
        return reverse('api:hh:token-refresh') + '?' + urlencode(params)

    @classmethod
    def _get_token(cls, user, grant_type, **kwargs):
        data = {
            'grant_type': grant_type,
            'redirect_uri': cls._get_redirect_uri(),
        }
        data.update(kwargs)

        oauth_data = cls._post('token', data)
        defaults = {
            'access_token': oauth_data['access_token'],
            'refresh_token': oauth_data['refresh_token'],
            'expiry_date': timezone.now() + timedelta(seconds=oauth_data['expires_in']),
        }
        token, created = HHOauthToken.objects.get_or_create(
            user=user,
            defaults=defaults,
        )
        if not created:
            for k, v in defaults.items():
                setattr(token, k, v)
            token.save()

        return token

    @classmethod
    def generate_token(cls, user, code):
        return cls._get_token(user, 'authorization_code', code=code)

    @classmethod
    def refresh_token(cls, token):
        return cls._get_token(token.user, 'refresh_token', refresh_token=token.refresh_token)
