from django.conf import settings

from intranet.femida.src.api.candidates.serializers import CandidateCreateVerificationFormSerializer
from intranet.femida.src.api.hire_orders.forms import (
    HireOrderCandidateForm,
    HireOrderMailForm,
    HireOrderOfferForm,
    HireOrderVacancyForm,
)
from intranet.femida.src.api.offers.serializers import OfferSendFormSerializer
from intranet.femida.src.core.models import Currency, City
from intranet.femida.src.offers.choices import WORK_PLACES
from intranet.femida.src.staff.models import Office
from intranet.femida.src.vacancies.choices import WAGE_SYSTEM_EXTENDED_TYPES, VACANCY_PRO_LEVELS


class HireOrderController:

    def __init__(self, instance):
        self.instance = instance

    @property
    def cleaned_candidate_data(self):
        assert not self.instance.is_incorrect
        validator = HireOrderCandidateForm(self.instance.raw_data['candidate'])
        assert validator.is_valid(), validator.errors_as_dict
        return validator.cleaned_data

    @property
    def cleaned_vacancy_data(self):
        assert not self.instance.is_incorrect
        validator = HireOrderVacancyForm(self.instance.raw_data['vacancy'])
        assert validator.is_valid(), validator.errors_as_dict
        data = validator.cleaned_data
        data['recruiters'] = [self.instance.recruiter]

        # Получаем дополнительные данные для вакансии из данных оффера
        offer_data = self.instance.raw_data['offer']
        data['wage_system'] = getattr(WAGE_SYSTEM_EXTENDED_TYPES, offer_data['payment_type'])
        pro_level = getattr(VACANCY_PRO_LEVELS, offer_data['professional_level'])
        data['pro_level_min'] = data['pro_level_max'] = pro_level
        data['max_salary'] = offer_data['salary']
        data['currency'] = Currency.objects.get(code=offer_data['payment_currency'])
        if offer_data['work_place'] == WORK_PLACES.office:
            data['cities'] = [Office.objects.get(id=offer_data['office']).city]
        else:
            data['cities'] = [City.objects.get(id=settings.CITY_HOMEWORKER_ID)]

        return data

    @property
    def cleaned_offer_data(self):
        validator = HireOrderOfferForm(self.instance.raw_data['offer'])
        assert validator.is_valid(), validator.errors_as_dict
        data = validator.cleaned_data
        return data

    def _get_cleaned_mail_data(self, offer_receiver=None):
        base_initial = {'offer_receiver': offer_receiver}
        validator = HireOrderMailForm(
            self.instance.raw_data.get('mail', {}),
            base_initial=base_initial,
        )
        assert validator.is_valid(), validator.errors_as_dict
        return validator.cleaned_data

    @property
    def cleaned_offer_send_data(self):
        offer = self.instance.offer
        data = OfferSendFormSerializer(offer).data
        mail_data = self._get_cleaned_mail_data(data['receiver'])
        data['bcc'] = [offer.boss] if offer.boss else []
        data['sender'] = mail_data['sender']
        data['receiver'] = mail_data['offer_receiver']
        if mail_data['offer_message']:
            data['message'] = mail_data['offer_message']
        return data

    @property
    def cleaned_verification_data(self):
        assert not self.instance.is_incorrect
        data = CandidateCreateVerificationFormSerializer(
            instance=self.instance.candidate,
            context={
                'user': self.instance.recruiter,
                'is_autohire': True,
            },
        ).data
        mail_data = self._get_cleaned_mail_data()
        if mail_data['sender']:
            data['sender'] = mail_data['sender']
        data['application'] = self.instance.application
        return data
