from typing import Dict

from intranet.femida.src.candidates.signals import (
    verification_failed,
    verification_on_check,
    verification_succeeded,
)
from intranet.femida.src.hire_orders.choices import (
    HIRE_ORDER_STATUSES as STATUSES,
    HIRE_ORDER_RESOLUTIONS as RESOLUTIONS,
)
from intranet.femida.src.hire_orders.models import HireOrder
from intranet.femida.src.hire_orders.tasks import perform_hire_order_action_task
from intranet.femida.src.offers.signals import (
    offer_confirmed,
    offer_accepted,
    offer_newhire_approved,
    offer_newhire_ready,
    offer_closed,
    offer_unapproved,
)
from intranet.femida.src.vacancies.signals import vacancy_approved, vacancy_unapproved


def action(name, **params) -> Dict:
    """
    Замыкание для запуска действий workflow на основе всех возможных сигналов.
    Отдаёт словарь с аргументами для присоединения к сигналу.
    """
    _supported_filter_params = {
        'candidate',
        'vacancy',
        'offer',
    }

    def inner(**filter_params):
        filter_params = {k: v for k, v in filter_params.items() if k in _supported_filter_params}
        hire_order_ids = (
            HireOrder.objects
            .exclude(status=STATUSES.closed)
            .filter(**filter_params)
            .values_list('id', flat=True)
        )
        for hire_order_id in hire_order_ids:
            perform_hire_order_action_task.delay(hire_order_id, name, **params)

    return {
        'receiver': inner,
        'dispatch_uid': f'hire_order_action_{name}',
        'weak': False,
    }


vacancy_approved.connect(**action('create_application'))
offer_confirmed.connect(**action('create_verification'))
verification_on_check.connect(**action('check_verification'))
verification_succeeded.connect(**action('send_offer'))
offer_accepted.connect(**action('accept_offer'))
offer_newhire_approved.connect(**action('approve_preprofile'))
offer_newhire_ready.connect(**action('finish_preprofile'))
offer_closed.connect(**action('close_offer'))

vacancy_unapproved.connect(**action('cancel', resolution=RESOLUTIONS.vacancy_unapproved))
offer_unapproved.connect(**action('cancel', resolution=RESOLUTIONS.offer_unapproved))
verification_failed.connect(**action('cancel', resolution=RESOLUTIONS.verification_failed))
