import logging

from collections import defaultdict
from contextlib import contextmanager
from functools import wraps

from django.conf import settings
from django.contrib.auth import get_user_model
from ylog.context import log_context

from intranet.femida.src.applications.controllers import create_application_gracefully
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.applications.workflow import ApplicationWorkflow
from intranet.femida.src.candidates.choices import (
    VERIFICATION_RESOLUTIONS,
    VERIFICATION_STATUSES,
    CONSIDERATION_RESOLUTIONS,
    CANDIDATE_STATUSES,
)
from intranet.femida.src.candidates.considerations.controllers import (
    get_or_create_relevant_consideration,
    get_relevant_consideration,
)
from intranet.femida.src.candidates.controllers import update_or_create_candidate
from intranet.femida.src.candidates.deduplication import DEFINITELY_DUPLICATE, MAYBE_DUPLICATE
from intranet.femida.src.candidates.deduplication.shortcuts import get_most_likely_duplicate
from intranet.femida.src.candidates.deduplication.strategies import new_strategy
from intranet.femida.src.candidates.helpers import (
    add_cities_from_vacancy,
    add_profession_from_vacancy,
)
from intranet.femida.src.candidates.models import Candidate, CandidateContact
from intranet.femida.src.candidates.verifications.helpers import is_verification_required
from intranet.femida.src.candidates.workflow import CandidateWorkflow
from intranet.femida.src.celery_app import NoRetry
from intranet.femida.src.communications.controllers import update_or_create_internal_message
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.signals import post_update, post_bulk_create
from intranet.femida.src.core.workflow import Action, Workflow
from intranet.femida.src.hire_orders.choices import HIRE_ORDER_STATUSES, HIRE_ORDER_RESOLUTIONS
from intranet.femida.src.hire_orders.controllers import HireOrderController
from intranet.femida.src.hire_orders.models import HireOrder
from intranet.femida.src.hire_orders.tasks import perform_hire_order_action_task
from intranet.femida.src.interviews.choices import APPLICATION_STATUSES, APPLICATION_RESOLUTIONS
from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.offers.workflow import OfferWorkflow
from intranet.femida.src.vacancies.controllers import update_or_create_vacancy
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.vacancies.workflow import VacancyWorkflow
from intranet.femida.src.utils.switches import is_candidate_main_recruiter_enabled


logger = logging.getLogger(__name__)

User = get_user_model()


def candidate_check(message):
    """
    Обёртка над проверками валидности кандидата.
    Подменяет результат проверки на заданный message и кэширует результат
    """
    def decorator(func):
        check_name = func.__name__

        @wraps(func)
        def wrapper(self, candidate):
            if not hasattr(self, '_cached_check_results'):
                self._cached_check_results = defaultdict(dict)
            candidate_checks = self._cached_check_results[candidate.id]
            if check_name not in candidate_checks:
                candidate_checks[check_name] = message if func(self, candidate) else None
            return candidate_checks[check_name]

        return wrapper
    return decorator


class InvalidCandidate(Exception):

    def __init__(self, errors):
        self.errors = errors

    @property
    def description(self):
        return '\n'.join(f'- {error}' for error in self.errors)


class HireOrderAction(Action):

    next_status = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.ctl = HireOrderController(self.instance)

    def is_status_correct(self):
        return super().is_status_correct() or self.must_be_skipped()

    def _is_already_done(self):
        if not self.next_status:
            return False
        # Разрешаем скипать экшны, если по истории мы уже совершали переход в этот `next_status`
        return self.instance.history.filter(status=self.next_status).exists()

    def must_be_skipped(self):
        return self._is_already_done() or self.instance.is_cancelled

    def get_log_context_data(self):
        return {
            'action_class': self.__class__.__name__,
            'hire_order': {
                'id': self.instance.id,
                'uuid': self.instance.uuid,
                'status': self.instance.status,
                'resolution': self.instance.resolution,
            },
            'candidate': self.instance.candidate_id,
            'application': self.instance.application_id,
            'vacancy': self.instance.vacancy_id,
            'offer': self.instance.offer_id,
        }

    @contextmanager
    def log_context(self, **kwargs):
        with log_context(**self.get_log_context_data(), **kwargs):
            yield

    def perform(self, **params):
        if self.must_be_skipped():
            return self.instance
        return self._perform(**params)


class HireOrderChangeStatusAction(HireOrderAction):

    current_status = None
    next_resolution = ''

    @property
    def valid_statuses(self):
        assert self.current_status is not None
        return [self.current_status]

    def _perform(self, **params):
        assert self.next_status is not None
        self.instance.status = self.next_status
        self.instance.resolution = self.next_resolution
        self.instance.save()
        return self.instance


class PrepareCandidateAction(HireOrderAction):

    valid_statuses = (HIRE_ORDER_STATUSES.new,)
    next_status = HIRE_ORDER_STATUSES.candidate_prepared

    def _perform(self, **params):
        candidate_data = self.ctl.cleaned_candidate_data
        duplicate, _, decision = get_most_likely_duplicate(
            candidate=candidate_data,
            strategy=new_strategy,
        )
        if decision == DEFINITELY_DUPLICATE:
            # Получаем кандидата из БД заново и блокируем строчку
            duplicate = Candidate.unsafe.select_for_update().get(id=duplicate.id)

        self.instance.candidate = duplicate
        try:
            self._validate_candidate(duplicate, decision)
        except InvalidCandidate as exc:
            self.instance.status = HIRE_ORDER_STATUSES.closed
            self.instance.resolution = HIRE_ORDER_RESOLUTIONS.invalid_candidate
            self.instance.resolution_description = exc.description
            self.instance.save()
            return self.instance

        if decision != DEFINITELY_DUPLICATE:
            self.instance.candidate = update_or_create_candidate(
                data=candidate_data,
                initiator=self.instance.recruiter,
            )
        else:
            self._update_contacts(candidate_data['contacts'])

        candidate_data = {
            'source': candidate_data['source'],
            'source_description': candidate_data.get('source_description', ''),
        }
        if is_candidate_main_recruiter_enabled():
            candidate_data['main_recruiter'] = self.instance.recruiter
        else:
            candidate_data['responsibles'] = [self.instance.recruiter]

        get_or_create_relevant_consideration(
            candidate=self.instance.candidate,
            initiator=self.instance.recruiter,
            candidate_data=candidate_data,
        )

        self.instance.status = self.next_status
        self.instance.save()
        self.wf.delay('create_vacancy')
        return self.instance

    def _validate_candidate(self, candidate, decision):
        if not candidate:
            return
        checks = [
            self._check_active_hire_orders,
            self._check_active_applications,
            self._check_nohire_verifications,
            self._check_active_offers,
            self._check_current_employee,
        ]
        if decision == MAYBE_DUPLICATE:
            checks.append(self._check_has_login)
        errors = (check(candidate) for check in checks)
        errors = [e for e in errors if e]
        with self.log_context():
            logger.log(
                logging.WARNING if errors else logging.INFO,
                '%s: validate candidate %d: %s',
                self.instance, candidate.id, errors,
            )
        if errors:
            raise InvalidCandidate(errors)

    @candidate_check('Кандидат уже проходит через процесс автонайма')
    def _check_active_hire_orders(self, candidate):
        active_hire_orders = (
            HireOrder.objects
            .filter(candidate=candidate)
            .exclude(status=HIRE_ORDER_STATUSES.closed)
        )
        return active_hire_orders.exists()

    @candidate_check('Кандидат претендует на другие вакансии')
    def _check_active_applications(self, candidate):
        return candidate.applications.filter(active_applications_query).exists()

    @candidate_check('Кандидат не прошёл проверку на конфликт интересов')
    def _check_nohire_verifications(self, candidate):
        return (
            candidate.verifications
            .alive()
            .filter(status=VERIFICATION_STATUSES.closed)
            .exclude(resolution=VERIFICATION_RESOLUTIONS.hire)
            .exists()
        )

    @candidate_check('У кандидата есть активный оффер')
    def _check_active_offers(self, candidate):
        return candidate.offers.alive().exists()

    @candidate_check('Кандидат является действующим сотрудником')
    def _check_current_employee(self, candidate):
        if not candidate.login:
            return False
        user = (
            User.objects
            .filter(
                username=candidate.login,
                is_dismissed=False,
            )
            .select_related('department')
            .first()
        )
        if not user:
            return False
        return user.department.is_in_trees((
            settings.YANDEX_DEPARTMENT_ID,
            settings.OUTSTAFF_DEPARTMENT_ID,
        ))

    @candidate_check('Есть похожий кандидат с логином')
    def _check_has_login(self, candidate):
        """
        Проверка похожа на _check_current_employee,
        но срабатывает только для похожих кандидатов, а не для 100% дублей.
        Проверяет, что мы не нанимаем сотрудника, для которого потом выяснится,
        что он у нас уже работал или числится внешним консультантом.
        """
        return not self._check_current_employee(candidate) and bool(candidate.login)

    def _update_contacts(self, main_contacts):
        """
        Обновляет контакты кандидата.
        Все переданные контакты становятся основными
        """
        candidate = self.instance.candidate
        main_contacts_dict = {(c['type'], c['normalized_account_id']): c for c in main_contacts}
        main_contacts_types = {c['type'] for c in main_contacts}
        existing_contacts = candidate.contacts.filter(is_active=True, type__in=main_contacts_types)
        contacts_to_update = []

        for contact in existing_contacts:
            key = (contact.type, contact.normalized_account_id)
            is_main = key in main_contacts_dict
            main_contacts_dict.pop(key, None)
            if contact.is_main != is_main:
                contact.is_main = is_main
                contacts_to_update.append(contact)

        CandidateContact.objects.bulk_update(contacts_to_update, fields=['is_main'])
        post_update.send(CandidateContact, queryset=contacts_to_update)

        contacts_to_create = [
            CandidateContact(candidate=candidate, **c)
            for c in main_contacts_dict.values()
        ]
        CandidateContact.objects.bulk_create(contacts_to_create)
        post_bulk_create.send(CandidateContact, queryset=contacts_to_create)


class CreateVacancyAction(HireOrderAction):

    valid_statuses = (HIRE_ORDER_STATUSES.candidate_prepared,)
    next_status = HIRE_ORDER_STATUSES.vacancy_on_approval

    def _perform(self, **params):
        vacancy_data = self.ctl.cleaned_vacancy_data
        vacancy = update_or_create_vacancy(
            data=vacancy_data,
            initiator=self.instance.recruiter,
        )
        self.instance.vacancy = vacancy
        self.instance.status = self.next_status
        self.instance.save()
        return self.instance


class CreateApplicationAction(HireOrderAction):

    valid_statuses = (HIRE_ORDER_STATUSES.vacancy_on_approval,)
    next_status = HIRE_ORDER_STATUSES.vacancy_prepared

    def _perform(self, **params):
        initiator = self.instance.recruiter
        application_data = dict(
            candidate=self.instance.candidate,
            vacancy=self.instance.vacancy,
            consideration=get_relevant_consideration(self.instance.candidate_id),
            status=APPLICATION_STATUSES.in_progress,
        )
        self.instance.application = create_application_gracefully(application_data, initiator)
        add_profession_from_vacancy(self.instance.candidate, self.instance.vacancy)
        add_cities_from_vacancy(self.instance.candidate, self.instance.vacancy)

        application_message = self.instance.raw_data.get('application_message')
        if application_message:
            message_data = dict(
                text=application_message,
                application=self.instance.application,
                candidate=self.instance.candidate,
            )
            update_or_create_internal_message(message_data, initiator, need_notify=False)

        self.instance.status = self.next_status
        self.instance.save()
        self.wf.delay('create_offer')
        return self.instance


class CreateOfferAction(HireOrderAction):

    valid_statuses = (HIRE_ORDER_STATUSES.vacancy_prepared,)
    next_status = HIRE_ORDER_STATUSES.offer_on_approval

    def _perform(self, **params):
        offer_data = self.ctl.cleaned_offer_data
        self.instance.offer = self.create_offer()
        self.instance.offer = update_instance(self.instance.offer, offer_data)

        self.approve_offer()
        self.instance.status = self.next_status
        self.instance.save()
        return self.instance

    def create_offer(self):
        workflow = ApplicationWorkflow(
            instance=self.instance.application,
            user=self.instance.recruiter,
        )
        return workflow.perform_action('create_offer', strict=True)

    def approve_offer(self):
        workflow = OfferWorkflow(self.instance.offer, user=self.instance.recruiter)
        workflow.perform_action('approve', strict=False)


class CreateVerificationAction(HireOrderAction):

    valid_statuses = (HIRE_ORDER_STATUSES.offer_on_approval,)
    next_status = HIRE_ORDER_STATUSES.verification_sent

    def _perform(self, **params):
        if not is_verification_required(self.instance.offer):
            self.wf.delay('send_offer')
            if not self.instance.force_verification_sending:
                return self.instance

        verification = self.instance.candidate.verifications.alive().first()
        verification_check_statuses = (
            VERIFICATION_STATUSES.on_check,
            VERIFICATION_STATUSES.on_ess_check,
        )
        if not verification:
            self.create_verification()
        elif verification.status in verification_check_statuses:
            self.wf.delay('check_verification')
        elif verification.status == VERIFICATION_STATUSES.closed:
            if verification.resolution != VERIFICATION_RESOLUTIONS.hire:
                self.raise_error('verification_resolution_negative', NoRetry)
            self.wf.delay('send_offer')

        self.instance.status = self.next_status
        self.instance.save()

        return self.instance

    def create_verification(self):
        candidate_workflow = CandidateWorkflow(self.instance.candidate, self.instance.recruiter)
        params = self.ctl.cleaned_verification_data
        return candidate_workflow.perform_action('create_verification', **params)


class CheckVerificationAction(HireOrderChangeStatusAction):

    current_status = HIRE_ORDER_STATUSES.verification_sent
    next_status = HIRE_ORDER_STATUSES.verification_on_check

    def must_be_skipped(self):
        return (
            super().must_be_skipped()
            or not self.instance.offer
            or not is_verification_required(self.instance.offer)
        )


class SendOfferAction(HireOrderAction):

    valid_statuses = (
        HIRE_ORDER_STATUSES.offer_on_approval,
        HIRE_ORDER_STATUSES.verification_sent,
        HIRE_ORDER_STATUSES.verification_on_check,
    )
    next_status = HIRE_ORDER_STATUSES.offer_sent

    def _perform(self, **params):
        send_data = self.ctl.cleaned_offer_send_data
        workflow = OfferWorkflow(self.instance.offer, self.instance.recruiter)
        workflow.perform_action('send', **send_data)
        self.instance.status = self.next_status
        self.instance.save()
        return self.instance


class AcceptOfferAction(HireOrderChangeStatusAction):

    current_status = HIRE_ORDER_STATUSES.offer_sent
    next_status = HIRE_ORDER_STATUSES.offer_accepted


class ApprovePreprofileAction(HireOrderChangeStatusAction):

    current_status = HIRE_ORDER_STATUSES.offer_accepted
    next_status = HIRE_ORDER_STATUSES.preprofile_approved


class FinishPreprofileAction(HireOrderChangeStatusAction):

    current_status = HIRE_ORDER_STATUSES.preprofile_approved
    next_status = HIRE_ORDER_STATUSES.preprofile_ready


class CloseOfferAction(HireOrderChangeStatusAction):

    current_status = HIRE_ORDER_STATUSES.preprofile_ready
    next_status = HIRE_ORDER_STATUSES.closed
    next_resolution = HIRE_ORDER_RESOLUTIONS.hired


class CancelAction(HireOrderAction):
    """
    Закрываем все сущности, связанные с заказом
    """
    def is_status_correct(self):
        return self.instance.status != HIRE_ORDER_STATUSES.closed

    def perform(self, resolution=HIRE_ORDER_RESOLUTIONS.cancelled, **params):
        self._delete_offer()
        self._close_vacancy()
        self._close_candidate()

        self.instance.status = HIRE_ORDER_STATUSES.closed
        self.instance.resolution = resolution
        self.instance.save()
        return self.instance

    def _delete_offer(self):
        """
        Удаляем оффер. Закрываем связанное претендентство и возвращаем вакансию в работу.
        """
        offer = self.instance.offer
        if not offer or offer.status in (OFFER_STATUSES.deleted, OFFER_STATUSES.rejected):
            return

        wf = OfferWorkflow(offer, self.instance.recruiter)
        wf.perform_action('delete_for_hire_order')

    def _close_vacancy(self):
        """
        Закрываем вакансию и, при наличии, связанное активное претендентство
        """
        vacancy = self.instance.vacancy
        if not vacancy or vacancy.status == VACANCY_STATUSES.closed:
            return

        wf = VacancyWorkflow(vacancy, self.instance.recruiter)
        wf.perform_action(
            action_name='close',
            application_resolution=APPLICATION_RESOLUTIONS.vacancy_closed,
            comment='Вакансия закрыта: отмена заказа на автоматизированный найм',
        )

    def _close_candidate(self):
        candidate = self.instance.candidate
        if not candidate or candidate.status == CANDIDATE_STATUSES.closed:
            return

        has_active_applications = candidate.applications.filter(active_applications_query).exists()
        if has_active_applications:
            with self.log_context():
                logger.warning(
                    'Can not close candidate %s because of other active applications',
                    candidate.id,
                )
            return

        wf = CandidateWorkflow(candidate, self.instance.recruiter)
        wf.perform_action(
            action_name='close',
            consideration_resolution=CONSIDERATION_RESOLUTIONS.no_offer,
        )


class HireOrderWorkflow(Workflow):

    ACTION_MAP = {
        'prepare_candidate': PrepareCandidateAction,
        'create_vacancy': CreateVacancyAction,
        'create_application': CreateApplicationAction,
        'create_offer': CreateOfferAction,
        'create_verification': CreateVerificationAction,
        'check_verification': CheckVerificationAction,
        'send_offer': SendOfferAction,
        'accept_offer': AcceptOfferAction,
        'approve_preprofile': ApprovePreprofileAction,
        'finish_preprofile': FinishPreprofileAction,
        'close_offer': CloseOfferAction,
        'cancel': CancelAction,
    }

    def delay(self, action_name):
        perform_hire_order_action_task.delay(self.instance.id, action_name)
