import json
import logging
import random
from typing import Generator

import waffle
from constance import config
from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.db.models import Case, Count, F, Q, Value, When
from django.utils.functional import cached_property

from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.candidates.models import Consideration
from intranet.femida.src.interviews.choices import (
    INTERVIEW_TYPES,
    INTERVIEW_GRADABLE_TYPES,
    INTERVIEW_STATES,
    INTERVIEW_RESOLUTIONS,
    AA_TYPES,
)
from intranet.femida.src.interviews.models import Interview, Application
from intranet.femida.src.staff.models import Department
from intranet.femida.src.users.utils import filter_users_by_aa_type
from intranet.femida.src.vacancies.choices import VACANCY_ROLES


User = get_user_model()

logger = logging.getLogger(__name__)


class InterviewVisibilityHelper:
    """
    Вычисляет видимость секций
    """
    UNBIASED_TYPES = (
        INTERVIEW_TYPES.regular,
        INTERVIEW_TYPES.aa,
    )

    def __init__(self, user, consideration, interviews=None):
        if interviews is None:
            interviews = list(consideration.interviews.all())
        self.user = user
        self.consideration = consideration
        self._interviews = interviews
        self.main_interviewer_ids = set()
        self.unfinished_main_interviewer_ids = set()
        self.unfinished_final_interviewer_ids = set()
        for i in self._interviews:
            if i.type in self.UNBIASED_TYPES:
                self.main_interviewer_ids.add(i.interviewer_id)
                if i.state in (Interview.STATES.assigned, Interview.STATES.estimated):
                    self.unfinished_main_interviewer_ids.add(i.interviewer_id)
            elif i.type == INTERVIEW_TYPES.final and i.state == Interview.STATES.assigned:
                self.unfinished_final_interviewer_ids.add(i.interviewer_id)

    @cached_property
    def is_unbiased_interviews_finished(self):
        return not self.unfinished_main_interviewer_ids

    @cached_property
    def is_user_allowed_to_see_all_finished_interviews(self):
        return (
            self.user.id not in self.unfinished_main_interviewer_ids
            and (
                self.user.id in self.main_interviewer_ids
                or self.user.id in self.unfinished_final_interviewer_ids
            )
        )

    @cached_property
    def is_visible_for_hiring_team(self):
        """
        Секции видны нанимающей команде, в зависимости от значения настройки
        INTERVIEW_VISIBILITY_FOR_HIRING_TEAM
        1. Если задан `default` – никаких привилегий у нанимающих нет
        2. Если `visible_if_no_interview` – секции видны,
           только если пользователь сам не является интервьюером этого кандидата
        3. Если `visible` – секции видны всегда
        """
        visibility = config.INTERVIEW_VISIBILITY_FOR_HIRING_TEAM
        if visibility == 'default':
            return False
        is_hiring_team_member = Application.unsafe.filter(
            active_applications_query,
            consideration_id=self.consideration.id,
            vacancy__memberships__member=self.user,
            vacancy__memberships__role__in=(
                VACANCY_ROLES.hiring_manager,
                VACANCY_ROLES.head,
                VACANCY_ROLES.responsible,
            ),
        ).exists()
        if not is_hiring_team_member:
            return False
        if visibility == 'visible_if_no_interview':
            return self.user.id not in self.unfinished_main_interviewer_ids
        if visibility == 'visible':
            return True

    def is_editable(self, interview):
        assert interview in self._interviews
        return (
            self.user.is_recruiter
            or self.user.id == interview.interviewer_id
        )

    def is_visible(self, interview):
        assert interview in self._interviews
        return (
            self.is_editable(interview)
            or self.user.is_recruiter_assessor
            or self.consideration.state == Consideration.STATES.archived
            or (
                interview.state == Interview.STATES.finished
                and (
                    interview.type not in self.UNBIASED_TYPES
                    or self.is_unbiased_interviews_finished
                    or self.is_user_allowed_to_see_all_finished_interviews
                    or self.is_visible_for_hiring_team
                )
            )
        )


def _get_reviewer_group_ids():
    try:
        mapping = json.loads(config.INTERVIEW_REVIEWER_GROUP_IDS)
        return {int(k): int(v) for k, v in mapping.items()}
    except (json.JSONDecodeError, ValueError, TypeError):
        logger.error(
            'Invalid constance config INTERVIEW_REVIEWER_GROUP_IDS: %s',
            config.INTERVIEW_REVIEWER_GROUP_IDS,
        )
        return {}


def can_user_see_review_issue(user, interview):
    if not interview.startrek_review_key:
        return False
    if interview.is_aa:
        return getattr(user, f'is_aa_{interview.aa_type}')

    return True


def get_reviewers_qs(queryset, interview):
    if interview.is_screening or interview.is_regular:
        group_id = settings.DEFAULT_INTERVIEW_REVIEWER_GROUP_ID
        profession_id = interview.application.vacancy.profession_id
        reviewer_group_ids = _get_reviewer_group_ids()
        if profession_id in reviewer_group_ids:
            group_id = reviewer_group_ids[profession_id]
        queryset = queryset.filter(groups__id=group_id)
    elif interview.is_aa:
        queryset = filter_users_by_aa_type(queryset, interview.aa_type)
    return queryset


def get_review_issue_assignee(interview, initiator=None):
    queryset = (
        User.objects
        .filter(is_dismissed=False)
        .exclude(id=interview.interviewer.id)
        .values_list('username', flat=True)
    )
    if initiator:
        queryset = queryset.exclude(id=initiator.id)
    reviewers = list(get_reviewers_qs(queryset, interview))
    return random.choice(reviewers) if reviewers else None


def is_interview_review_needed(interview: Interview):
    if interview.is_screening:
        interview_prof_sphere = interview.application.vacancy.professional_sphere_id
        if interview_prof_sphere == settings.DEVELOPMENT_PROF_SPHERE_ID:
            return (
                random.randrange(100) < config.REVIEW_SCREENING_PROBABILITY
                or interview.is_finished_by_newbie
            )

    elif interview.is_regular:
        interview_prof_sphere = interview.application.vacancy.professional_sphere_id
        if interview_prof_sphere == settings.DEVELOPMENT_PROF_SPHERE_ID:
            return interview.is_finished_by_newbie

    elif interview.is_aa and not interview.is_aa_code17plus:
        probability_conf = json.loads(config.REVIEW_AA_TYPE_PROBABILITY)
        if interview.aa_type in probability_conf:
            return random.randrange(100) < probability_conf.get(interview.aa_type)
        return True

    return False


def get_interview_direction_ids(qs):
    """
    Отдаёт множество ID отделов-направлений, в которых проводились секции
    """
    department_ids = set(qs.values_list('application__vacancy__department_id', flat=True))
    departments = (
        Department.objects
        .with_direction()
        .filter(id__in=department_ids)
    )
    return {d.direction_id for d in departments}


def add_calculated_resolution_to_qs(qs):
    """
    Добавляет к QuerySet'у секций вычисленную резолюцию.
    Сейчас резолюция бывает только у hr-скринингов и финалов.
    При этом, просто начать использовать резолюцию для всех типов секций нельзя
    – как минимум, есть проблема на фронте
    """
    return qs.annotate(
        calculated_resolution=Case(
            When(
                type__in=INTERVIEW_GRADABLE_TYPES._db_values,
                grade=0,
                then=Value(INTERVIEW_RESOLUTIONS.nohire),
            ),
            When(
                type__in=INTERVIEW_GRADABLE_TYPES._db_values,
                grade__gt=0,
                then=Value(INTERVIEW_RESOLUTIONS.hire),
            ),
            default=F('resolution'),
        ),
    )


def is_unfinished_interview_exists(**filter_params) -> bool:
    return (
        Interview.unsafe
        .exclude(state__in=(
            INTERVIEW_STATES.finished,
            INTERVIEW_STATES.cancelled,
        ))
        .filter(**filter_params)
        .exists()
    )


def get_enabled_aa_types() -> Generator:
    return (
        i for i, _ in AA_TYPES
        if not waffle.switch_is_active(f'disable_aa_{i}_type')
    )


def get_aa_types_count():
    """
    Получаем количество собеседующих для каждой аа группы
    """
    group_ids = settings.AA_TYPE_TO_GROUP_ID.values()
    groups = (
        Group.objects
        .filter(id__in=group_ids)
        .annotate(count=Count('user', filter=Q(user__is_dismissed=False)))
        .values('id', 'count')
    )
    counter = {
        group['id']: group['count']
        for group in groups
    }
    result = {
        aa_type: counter.get(settings.AA_TYPE_TO_GROUP_ID[aa_type], 0)
        for aa_type in settings.AA_TYPE_TO_GROUP_ID
    }
    return result
