import waffle

from constance import config
from django.conf import settings
from django.contrib.contenttypes.fields import GenericRelation
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from model_utils.models import TimeStampedModel
from model_utils.choices import Choices
from model_utils.fields import AutoLastModifiedField

from intranet.femida.src.core.db.fields import StartrekIssueKeyField
from intranet.femida.src.core.switches import TemporarySwitch
from intranet.femida.src.interviews.managers import InterviewManager
from intranet.femida.src.permissions.managers.candidate import CandidatePermManager
from intranet.femida.src.permissions.managers.interview import InterviewPermManager
from intranet.femida.src.candidates.models import Candidate, CandidateSubmission
from intranet.femida.src.interviews import managers
from intranet.femida.src.vacancies.models import Vacancy
from intranet.femida.src.wf.models import WFModelMixin

from . import choices


class Application(TimeStampedModel):

    unsafe = models.Manager()
    objects = CandidatePermManager(perm_prefix='candidate')

    modified = AutoLastModifiedField(_('modified'), db_index=True)

    candidate = models.ForeignKey(
        to=Candidate,
        on_delete=models.CASCADE,
        related_name='applications',
    )
    vacancy = models.ForeignKey(
        to=Vacancy,
        on_delete=models.CASCADE,
        related_name='applications',
    )
    consideration = models.ForeignKey(
        to='candidates.Consideration',
        on_delete=models.CASCADE,
        related_name='applications',
        null=True,
    )

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        null=True,
    )
    status = models.CharField(
        max_length=32,
        choices=choices.APPLICATION_STATUSES,
        default=choices.APPLICATION_STATUSES.draft,
    )
    resolution = models.CharField(
        max_length=64,
        choices=choices.APPLICATION_RESOLUTIONS,
        default='',
        blank=True,
    )
    status_changed = models.DateTimeField(auto_now_add=True)

    submission = models.ForeignKey(
        to=CandidateSubmission,
        on_delete=models.CASCADE,
        related_name='applications',
        null=True,
        blank=True,
    )
    rank = models.SmallIntegerField(default=0)
    # TODO: deprecated-поле: удалить. Сейчас сюда дублируется комментарий при создании APL
    comment = models.TextField(default='', blank=True)
    source = models.CharField(
        max_length=16,
        choices=choices.APPLICATION_SOURCES,
        default=choices.APPLICATION_SOURCES.recruiter,
    )
    proposal_status = models.CharField(
        max_length=32,
        choices=choices.APPLICATION_PROPOSAL_STATUSES,
        default=choices.APPLICATION_PROPOSAL_STATUSES.undefined,
        blank=True,
        null=True,
    )
    proposal_factors = JSONField(null=True, blank=True)
    _is_archived = None

    @property
    def is_archived(self):
        if self._is_archived is None:
            self._is_archived = self.status == choices.APPLICATION_STATUSES.closed
        return self._is_archived

    @is_archived.setter
    def is_archived(self, value):
        self._is_archived = value

    def __str__(self):
        return 'Application {}, vacancy {}, candidate {}'.format(
            self.id,
            self.vacancy_id,
            self.candidate_id,
        )

    def __repr__(self):
        return 'Application {}'.format(self.id)

    class Meta:
        default_manager_name = 'unsafe'
        indexes = [
            models.Index(
                fields=['id', 'modified'],
                name='application_id_modified_idx',
            ),
        ]


class Assignment(WFModelMixin, TimeStampedModel):
    """
    Назначение. Привзяка выбранной задачи к конкретной секции
    """
    objects = managers.AssignmentManager()

    WIKI_FIELDS_MAP = {
        'comment': 'formatted_comment',
    }

    GRADES = Choices(
        (1, 'BAD', 'Плохо'),
        (3, 'NORMAL', 'Норм'),
        (5, 'EXCELENT', 'Отлично'),
    )

    problem = models.ForeignKey(
        to='problems.Problem',
        on_delete=models.CASCADE,
        related_name='assignments',
    )
    interview = models.ForeignKey(
        to='interviews.Interview',
        on_delete=models.CASCADE,
        related_name='assignments',
    )

    comment = models.TextField(
        default='',
        blank=True,
    )
    formatted_comment = models.TextField(
        default='',
        blank=True,
    )

    grade = models.SmallIntegerField(
        choices=GRADES,
        null=True,
        blank=True,
    )
    position = models.IntegerField(default=0)

    class Meta:
        ordering = ['position']

    def __str__(self):
        return 'Assignment {}, problem {}, interview {}'.format(
            self.id,
            self.problem_id,
            self.interview_id,
        )


class Interview(WFModelMixin, TimeStampedModel):

    unsafe = InterviewManager()
    objects = InterviewPermManager()

    WIKI_FIELDS_MAP = {
        'comment': 'formatted_comment',
    }

    # TODO: Вынести в choices.py
    GRADES = (
        (0, 'Отказать'),
        (1, 'Нанять, I Стажёр'),
        (2, 'Нанять, II Младший специалист'),
        (3, 'Нанять, III Специалист'),
        (4, 'Нанять, IV Специалист'),
        (5, 'Нанять, V Старший специалист'),
        (6, 'Нанять, VI Ведущий специалист'),
        (7, 'Нанять, VII Ведущий специалист'),
        (8, 'Нанять, VIII Эксперт'),
    )
    STATES = Choices(
        (choices.INTERVIEW_STATES.draft, 'Черновик'),
        (choices.INTERVIEW_STATES.assigned, 'Назначена'),
        (choices.INTERVIEW_STATES.estimated, 'Оценена'),
        (choices.INTERVIEW_STATES.finished, 'Завершена'),
        (choices.INTERVIEW_STATES.cancelled, 'Отменена'),
    )

    type = models.CharField(
        max_length=16,
        choices=choices.INTERVIEW_TYPES,
        default=choices.INTERVIEW_TYPES.regular,
        null=True,
    )

    application = models.ForeignKey(
        to='interviews.Application',
        on_delete=models.CASCADE,
        related_name='interviews',
        blank=True,
        null=True,
    )
    candidate = models.ForeignKey(
        to='candidates.Candidate',
        on_delete=models.CASCADE,
        related_name='interviews',
        blank=True,
        null=True,
    )
    interviewer = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='interviews',
        null=True,
        blank=True,
    )
    optional_participant = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='optional_interviews',
        null=True,
        blank=True,
    )
    state = models.CharField(
        max_length=32,
        choices=STATES,
        default='assigned',
    )
    resolution = models.CharField(
        max_length=32,
        choices=choices.INTERVIEW_RESOLUTIONS,
        default='',
        blank=True,
        null=True,
    )
    problems = models.ManyToManyField(
        to='problems.Problem',
        through='Assignment',
        related_name='interviews',
    )
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='created_interviews',
    )
    finished = models.DateTimeField(
        null=True,
        blank=True,
    )
    finished_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='finished_interviews',
        null=True,
        blank=True,
    )

    round = models.ForeignKey(
        to='interviews.InterviewRound',
        on_delete=models.PROTECT,
        related_name='interviews',
        null=True,
        blank=True,
    )
    potential_interviewers = models.ManyToManyField(
        to=settings.AUTH_USER_MODEL,
        related_name='+',
        blank=True,
    )

    comment = models.TextField(
        default='',
        blank=True,
    )
    formatted_comment = models.TextField(
        default='',
        blank=True,
    )

    cancel_reason = models.TextField(
        blank=True,
    )
    grade = models.SmallIntegerField(
        choices=GRADES,
        null=True,
        blank=True,
    )
    # Пока добавляем просто флаг, чтобы отличать оценки старой шкалы и новой
    # После того, как все устаканиться и мы поймем будем ли мы мигрировать
    # старые оценки в новые и т.п., придумаем более правильное решение
    is_pro_level_scale = models.BooleanField(
        blank=True,
        default=True,
    )
    event_id = models.IntegerField(
        null=True,
        blank=True,
    )
    event_start_time = models.DateTimeField(
        null=True,
        blank=True,
    )
    # хотя поле и является временем, но по факту - это строка-ключ,
    # которая используется совместно с event_id
    # для получения конкретного события из цепочки повторяющихся
    event_instance_start_ts = models.CharField(
        max_length=255,
        blank=True,
        null=True,
    )
    section = models.CharField(
        max_length=255,
        blank=True,
        default='',
    )
    sent = models.BooleanField(
        blank=True,
        default=False,
    )
    preset = models.ForeignKey(
        to='problems.Preset',
        on_delete=models.CASCADE,
        related_name='interviews',
        blank=True,
        null=True,
    )
    consideration = models.ForeignKey(
        to='candidates.Consideration',
        on_delete=models.CASCADE,
        related_name='interviews',
        blank=True,
        null=True,
    )

    aa_type = models.CharField(
        max_length=32,
        choices=choices.AA_TYPES,
        blank=True,
        null=True,
    )
    # With code on board or not
    is_code = models.BooleanField(blank=True, default=False)

    # Тикет AAREV в трекере
    startrek_review_key = StartrekIssueKeyField(null=True, blank=True)

    comments = GenericRelation('comments.Comment')

    yandex_grade = models.SmallIntegerField(
        choices=choices.INTERVIEW_YANDEX_GRADES,
        null=True,
        blank=True,
        help_text='Оценка по новой шкале (настоящий грейд)',
    )
    scale = models.CharField(
        max_length=16,
        choices=choices.INTERVIEW_SCALES,
        null=True,
        blank=True,
        help_text='Предпочтительная шкала оценки',
    )

    # Флаг для определения новой формы создания интервью
    is_new_form = models.NullBooleanField(blank=True)

    def get_grade_display(self):
        if self.grade is None:
            return choices.INTERVIEW_RESOLUTIONS_TRANSLATIONS.get(self.resolution)
        else:
            return self.GRADES[self.grade][1]

    def get_grade_display_en(self):
        if self.grade is None:
            return choices.INTERVIEW_RESOLUTIONS_TRANSLATIONS_EN.get(self.resolution)
        else:
            return choices.INTERVIEW_HIRE_GRADES_TRANSLATIONS_EN.get(self.grade)

    @property
    def is_alive(self):
        return self.state in choices.INTERVIEW_ALIVE_STATES

    @property
    def is_hire(self):
        """
        Отвечает на вопрос, положительный или нет результат секции для кандидата.
        Если нет ни грейда, ни резолюции, то считаем, что ответ неопределен (None).
        """
        if self.grade is not None:
            return self.grade > 0
        if self.resolution:
            return self.resolution == choices.INTERVIEW_RESOLUTIONS.hire

    @property
    def is_screening(self):
        return self.type == choices.INTERVIEW_TYPES.screening

    @property
    def is_regular(self):
        return self.type == choices.INTERVIEW_TYPES.regular

    @property
    def is_aa(self):
        return self.type == choices.INTERVIEW_TYPES.aa

    @property
    def is_gradable(self):
        return self.type in choices.INTERVIEW_GRADABLE_TYPES

    @property
    def event_url(self):
        if self.event_id:
            event_url = '{base_url}event/?event_id={event_id}'.format(
                base_url=settings.CALENDAR_URL,
                event_id=self.event_id,
            )
            # TODO: удалить свитч и старую реализацию после релиза FEMIDA-7359
            if (
                self.event_instance_start_ts
                and waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_EVENT_URL_FORMAT)
            ):
                # добавляем поддержку периодических встреч. Пример:
                # https://calendar.testing.yandex-team.ru/event/2801388?event_date=2022-06-23T09%3A00%3A00
                event_url += f'&event_date={self.event_instance_start_ts}'
            return event_url

    @cached_property
    def is_finished_by_newbie(self):
        """
        Определяет, входит ли текущая секция в число первых N секций собеседующего,
        которые необходимо принудительно отправлять на ревью.
        """
        assert self.state == self.STATES.finished
        # Кол-во секций собеседующего складывается из всех завершенных им gradable секций
        interviews_finished = (
            Interview.unsafe
            .gradable()
            .filter(
                interviewer_id=self.interviewer_id,
                state=self.STATES.finished,
            )
            .count()
        )
        return interviews_finished <= config.INTERVIEWER_NEWBIE_THRESHOLD

    def __getattr__(self, item):
        if item.startswith('is_aa_'):
            aa_type = item[6:]
            if aa_type in choices.AA_TYPES:
                return self.aa_type == aa_type
        raise AttributeError(f"'Interview' object has no attribute '{item}'")

    def __str__(self):
        return 'Interview {}, consideration {}, candidate {}'.format(
            self.id,
            self.consideration_id,
            self.candidate_id,
        )

    class Meta:
        default_manager_name = 'unsafe'


class InterviewRound(TimeStampedModel):

    candidate = models.ForeignKey(
        to='candidates.Candidate',
        on_delete=models.CASCADE,
        related_name='+',
    )
    consideration = models.ForeignKey(
        to='candidates.Consideration',
        on_delete=models.CASCADE,
        related_name='interview_rounds',
    )
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='+',
    )

    type = models.CharField(max_length=16, choices=choices.INTERVIEW_ROUND_TYPES)
    status = models.CharField(
        max_length=32,
        choices=choices.INTERVIEW_ROUND_STATUSES,
        default=choices.INTERVIEW_ROUND_STATUSES.new,
    )
    planner = models.CharField(
        max_length=16,
        choices=choices.INTERVIEW_ROUND_PLANNERS,
        blank=True,
    )

    office = models.ForeignKey(
        to='staff.Office',
        on_delete=models.PROTECT,
        related_name='+',
        blank=True,
        null=True,
    )
    is_strict_order = models.BooleanField(default=False, help_text='Порядок секций важен?')
    lunch_duration = models.IntegerField(
        choices=choices.INTERVIEW_ROUND_LUNCH_DURATIONS,
        default=choices.INTERVIEW_ROUND_LUNCH_DURATIONS.no_lunch,
        help_text='Время перерыва на обед в минутах',
    )
    comment = models.TextField(
        default='',
        blank=True,
        null=True,
        help_text='Комментарий для асессоров',
    )
    message = models.ForeignKey(
        to='communications.Message',
        on_delete=models.SET_NULL,
        related_name='+',
        null=True,
        blank=True,
        help_text='Письмо-приглашение кандидату',
    )

    def __str__(self):
        return (
            f'Interview {self.type} round {self.id}, '
            f'consideration {self.consideration_id}, candidate {self.candidate_id}'
        )


class InterviewRoundTimeSlot(models.Model):
    """
    Слот времени, в который кандидату было бы удобно пройти интервью
    """
    round = models.ForeignKey(
        to=InterviewRound,
        on_delete=models.CASCADE,
        related_name='time_slots',
    )
    start = models.DateTimeField()
    end = models.DateTimeField()
