import logging

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError
from django.core.mail.message import make_msgid
from django.db import transaction
from django.template.loader import render_to_string
from django.utils import translation
from ylog.context import log_context

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.candidates.helpers import is_russian_speaking_candidate
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.celery_app import app, NoRetry
from intranet.femida.src.interviews.models import Interview, InterviewRound
from intranet.femida.src.interviews.startrek.issues import (
    create_review_issue,
    create_aa_entry_issue,
)
from intranet.femida.src.interviews.choices import INTERVIEW_TYPES, AA_TYPES
from intranet.femida.src.notifications.utils import get_base_context
from intranet.femida.src.tasks import send_email


logger = logging.getLogger(__name__)


User = get_user_model()


# TODO: удалить параметр assistants после релиза FEMIDA-5559
@app.autoretry_task(max_retries=3)
def create_review_issue_task(interview_id, queue,
                             initiator_id=None, assignee=None, assistants=None):
    interview = Interview.unsafe.get(id=interview_id)
    if interview.startrek_review_key:
        logger.warning(
            'Review issue was already created for interview %s: %s',
            interview.id,
            interview.startrek_review_key,
        )
        return
    initiator = User.objects.get(id=initiator_id) if initiator_id else None
    create_review_issue(
        interview=interview,
        queue=queue,
        initiator=initiator,
        assignee=assignee,
    )


@app.autoretry_task(max_retries=3)
def create_aa_entry_issue_task(applicant_id, answers):
    applicant = User.objects.get(id=applicant_id)
    issue = create_aa_entry_issue(applicant, answers)
    return issue.key


@app.autoretry_task(max_retries=3)
def send_interview_survey_task(consideration_id, was_offer_rejected=False):
    """
    Шлём кандидату опрос по найму, если он прошёл дальше HR-скрининга
    """
    from intranet.femida.src.candidates.helpers import get_main_email

    if not is_russian_speaking_candidate(consideration_id):
        return

    interviews = (
        Interview.unsafe
        .filter(
            consideration=consideration_id,
            state=Interview.STATES.finished,
        )
        .exclude(type=INTERVIEW_TYPES.hr_screening)
        .select_related('application__vacancy__department')
        .prefetch_related('problems')
    )
    if not interviews:
        return
    candidate = Candidate.unsafe.get(considerations=consideration_id)
    email = get_main_email(candidate)
    if not email:
        return

    context = get_base_context()
    context['candidate_id'] = candidate.id
    context['candidate_first_name'] = candidate.first_name
    is_full_survey = _get_is_full_survey(interviews)
    context['interviews'] = interviews if is_full_survey else []
    context['survey_id'] = (
        settings.INTERVIEW_FULL_SURVEY_ID
        if is_full_survey
        else settings.INTERVIEW_SHORT_SURVEY_ID
    )
    context['was_offer_rejected'] = was_offer_rejected

    translation.activate('ru')
    body = render_to_string(
        template_name='email/interviews/candidate-survey.html',
        context=context,
    )
    translation.deactivate()

    # Логируем контекст при отправке опроса, чтобы можно было строить аналитику по этим данным
    survey_log_context = {
        'candidate_id': candidate.id,
        'consideration_id': consideration_id,
        'survey_id': context['survey_id'],
        'was_offer_rejected': was_offer_rejected,
    }
    with log_context(interview_survey=survey_log_context):
        logger.info('Sending interview survey to candidate %s', candidate.id)

    send_email.delay(
        subject='Как прошло ваше интервью в Яндексе?',
        body=body,
        to=[email],
        reply_to=settings.JOB_EMAIL_VERBOSE,
        from_email=settings.JOB_EMAIL_VERBOSE,
        message_id=make_msgid(),
        is_external=True,
    )


def _get_is_full_survey(interviews):
    """
    Определеяет тип опроса по критериям из FEMIDA-3612
    """
    for i in interviews:
        if i.is_aa:
            if i.aa_type == AA_TYPES.management:
                return False
            continue
        v = i.application.vacancy
        is_short_survey = (
            v.type == VACANCY_TYPES.internship
            or v.professional_sphere_id != settings.DEVELOPMENT_PROF_SPHERE_ID
            or v.department is None
        )
        if is_short_survey:
            return False
    return True


@app.autoretry_task(max_retries=3)
@transaction.atomic
@action_logged('interview_round_finish_planning')
def interview_round_finish_planning_task(interview_round_id, data):
    from intranet.femida.src.interviews.yang.controllers import interview_round_finish_planning
    interview_round = (
        InterviewRound.objects
        .select_for_update(of=('self',))
        .select_related(
            'created_by',
            'candidate',
            'consideration',
            'message',
        )
        .get(id=interview_round_id)
    )
    try:
        interview_round_finish_planning(interview_round, data)
    except ValidationError as exc:
        raise NoRetry(exc)
