import sform

from intranet.femida.src.api.core.forms import UserSuggestField
from intranet.femida.src.calendar import get_event
from intranet.femida.src.calendar.exceptions import EventDoesNotExist, EventIsNotAvailable
from intranet.femida.src.core.exceptions import SimpleValidationError
from intranet.femida.src.interviews.choices import INTERVIEW_STATES, INTERVIEW_ROUND_STATUSES


def flat_choices(*choices):
    return [(c, c) for c in choices]


def validate_aa_interviewer(interview, interviewer):
    if not interviewer.is_aa:
        raise SimpleValidationError('interviewer_not_aa')
    if not getattr(interviewer, f'is_aa_{interview.aa_type}', False):
        raise SimpleValidationError('interviewer_wrong_aa')


class YangInterviewForm(sform.SForm):

    id = sform.IntegerField(state=sform.REQUIRED)
    state = sform.ChoiceField(
        state=sform.REQUIRED,
        choices=flat_choices(
            INTERVIEW_STATES.assigned,
            INTERVIEW_STATES.cancelled,
        ),
    )
    interviewer = UserSuggestField()
    event_id = sform.IntegerField()

    @property
    def interviews_by_id(self):
        return self.base_initial['interviews_by_id']

    def clean_id(self, value):
        if value not in self.interviews_by_id:
            raise SimpleValidationError('invalid')
        return value

    def clean_interviewer(self, interviewer):
        if not interviewer:
            return interviewer
        interview_id = self.cleaned_data.get('id')
        interview = self.interviews_by_id.get(interview_id)
        if interview and interview.is_aa:
            validate_aa_interviewer(interview, interviewer)
        return interviewer

    def get_field_state(self, name):
        if name in ('interviewer', 'event_id'):
            state = self.cleaned_data.get('state')
            return sform.REQUIRED if state == INTERVIEW_STATES.assigned else sform.READONLY
        return super().get_field_state(name)

    def clean(self):
        cleaned_data = super().clean()
        event_id = cleaned_data.get('event_id')
        if event_id:
            try:
                event = get_event(event_id, strict=True)
            except (EventDoesNotExist, EventIsNotAvailable):
                raise SimpleValidationError('event_not_available')
            cleaned_data['event'] = event
            cleaned_data['event_start_time'] = event.start_time
        return cleaned_data


class YangInterviewRoundForm(sform.SForm):

    interviews = sform.GridField(sform.FieldsetField(YangInterviewForm))

    def clean_interviews(self, interviews):
        interviews = interviews or []
        interview_ids = {i['id'] for i in interviews}
        valid_interview_ids = set(self.base_initial['interviews_by_id'])
        if interview_ids != valid_interview_ids:
            raise SimpleValidationError('invalid')
        return interviews

    def clean(self):
        cleaned_data = super().clean()
        interviews = self.cleaned_data.get('interviews') or []
        cleaned_data['status'] = (
            INTERVIEW_ROUND_STATUSES.cancelled
            if all(i['state'] == INTERVIEW_STATES.cancelled for i in interviews)
            else INTERVIEW_ROUND_STATUSES.planned
        )
        return cleaned_data
