import pytz
from collections import OrderedDict

import waffle

from dateutil.relativedelta import relativedelta
from django.contrib.auth import get_user_model
from django.utils import timezone

from intranet.femida.src.candidates.contacts import normalize_contact
from intranet.femida.src.candidates.choices import (
    CANDIDATE_STATUSES,
    RKN_CONTACT_TYPES,
    CANDIDATE_SORTING_TYPES,
)
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.utils.datetime import to_timestamp
from intranet.femida.src.utils.translation import get_name_field, get_localized_name_field
from .api import make_api_request


User = get_user_model()


class SearchResultsCollection:
    """
    Класс, который можно передать в Paginator, например

    TODO: Надо порефакторить. Какой-то несносный класс получился в итоге.
    """
    scope = None

    def __init__(self, text, user_ticket, ip, filter_params=None, user=None):
        self.text = text
        self.user_ticket = user_ticket
        self.ip = ip
        self.filter_params = filter_params or {}
        self.user = user
        self._count = None
        self._request_url = None

    def prepare_filter_params(self, data):
        return data

    def fetch_results(self, data):
        return data

    def _get_search_results(self, page=None):
        params = OrderedDict(
            scope=self.scope,
            format='json',
            p=page - 1 if page is not None else 0,
            text=self.text,
            allow_empty=1,
        )
        if self.filter_params is not None:
            params.update(self.prepare_filter_params(self.filter_params))

        response = make_api_request(params, self.user_ticket, self.ip)
        self._request_url = response.url
        return response.json()

    def get_page(self, page):
        search_results = self._get_search_results(page=page)
        self._count = search_results['meta']['count']
        docs = [i['snippet'] for i in search_results['results']['docs']]
        return self.fetch_results(docs)

    def count(self):
        if self._count is not None:
            return self._count
        search_results = self._get_search_results()
        self._count = search_results['meta']['count']
        return self._count

    def __len__(self):
        return self.count()


class CandidateSearchCollection(SearchResultsCollection):

    scope = 'candidatesearch'

    def _get_facets(self, data):
        result = OrderedDict()
        if 'professions' in data:
            result['facet.profession'] = [p.id for p in data['professions']]
        if 'skills' in data:
            result['facet.skill'] = [s.id for s in data['skills']]
        if 'responsibles' in data:
            result['facet.responsible'] = [s.username for s in data['responsibles']]
        if 'target_cities' in data:
            result['facet.target_city'] = [c.id for c in data['target_cities']]
        if 'tags' in data:
            result['facet.tag'] = [t.name for t in data['tags']]
        if data.get('is_active') is not None:
            active_statuses_map = {
                True: CANDIDATE_STATUSES.in_progress,
                False: CANDIDATE_STATUSES.closed,
            }
            result['facet.status'] = active_statuses_map.get(data['is_active'])
        return result

    def _get_attrs(self, data):
        result = OrderedDict()
        if data.get('without_nohire', False):
            result['attr.nohire_interviews_count'] = 0
            result['attr.hire_interviews_count__gte'] = 1
        if 'skype_interviews_avg_grade' in data:
            result['attr.skype_interviews_avg_grade__gte'] = data['skype_interviews_avg_grade']
        if 'on_site_interviews_avg_grade' in data:
            result['attr.on_site_interviews_avg_grade__gte'] = data['on_site_interviews_avg_grade']
        if 'hire_interviews_avg_grade' in data:
            result['attr.hire_interviews_avg_grade__gte'] = data['hire_interviews_avg_grade']
        if data.get('ignore_employees'):
            result['attr.is_current_employee'] = 0
        for date_param in ['created__gte', 'created__lte', 'modified__gte', 'modified__lte']:
            _date = data.get(date_param)
            if _date:
                if 'lte' in date_param:
                    _time = timezone.datetime.max.time()
                else:
                    _time = timezone.datetime.min.time()
                dt = timezone.datetime.combine(_date, _time)
                if self.user and self.user.timezone:
                    dt = pytz.timezone(self.user.timezone).localize(dt)
                result['attr.{}'.format(date_param)] = to_timestamp(dt)
        return result

    def _get_zones(self, data):
        result = OrderedDict()
        for field_name in ['city', 'institution', 'employer']:
            if field_name in data:
                result['zone.{}'.format(field_name)] = data[field_name]
        return result

    def _get_order(self, data):
        sort = data.get('sort', CANDIDATE_SORTING_TYPES.relevance)
        result = OrderedDict()
        if sort == CANDIDATE_SORTING_TYPES.modified_asc:
            result['sorted'] = 'updated'
            result['sorted_order'] = 'asc'
        elif sort == CANDIDATE_SORTING_TYPES.modified_desc:
            result['sorted'] = 'updated'
            result['sorted_order'] = 'desc'
        else:
            result['sorted'] = 'rlv'
        return result

    def prepare_filter_params(self, data):
        result = OrderedDict()
        result.update(self._get_facets(data))
        result.update(self._get_attrs(data))
        result.update(self._get_zones(data))
        result.update(self._get_order(data))

        is_recruiter = self.user.is_recruiter or self.user.is_recruiter_assessor
        if is_recruiter and waffle.switch_is_active('is_rkn'):
            result['attr.modified__gte'] = to_timestamp(timezone.now() - relativedelta(years=5))

        return result

    def _replace_usernames_with_user_objects(self, data):
        usernames = []
        for item in data:
            for vacancy in item['vacancies']:
                usernames.extend(vacancy['access'])
            usernames.extend(item['responsibles'])

        users = (
            User.objects
            .filter(username__in=set(usernames))
            .values('id', 'username', 'first_name', 'last_name')
        )
        user_map = {item['username']: item for item in users}

        for item in data:
            for vacancy in item['vacancies']:
                vacancy['access'] = [user_map[username] for username in vacancy['access']]
            item['responsibles'] = [user_map[username] for username in item['responsibles']]

    def _merge_target_city_names(self, data):
        field_name = get_name_field()
        for item in data:
            for city in item['target_cities']:
                city['name'] = city[field_name]

    def _localize_profession_names(self, data):
        field_name = get_localized_name_field()
        if field_name == 'name':
            return
        for item in data:
            for profession in item['professions']:
                profession['name'] = profession.get(field_name) or profession['name']

    def _normalize_account_ids(self, data):
        for item in data:
            for contact in item['contacts']:
                normalized_account_id = normalize_contact(
                    contact_type=contact['type'],
                    account_id=contact['account_id'],
                )
                if normalized_account_id:
                    contact['account_id'] = normalized_account_id

    def _set_is_available(self, data):
        """
        Поскольку выдача из поиска шире, чем из базы, добавим флаг is_available каждому кандидату
        """
        cand_ids = [item['id'] for item in data]
        available_cand_ids = set(
            Candidate.objects.filter(id__in=cand_ids).values_list('id', flat=True)
        )
        for item in data:
            item['is_available'] = item['id'] in available_cand_ids

    def fetch_results(self, data):
        for item in data:
            item['target_cities'] = item['target_cities'] or []
            item['tags'] = item['tags'] or []
        self._replace_usernames_with_user_objects(data)
        self._merge_target_city_names(data)
        self._localize_profession_names(data)

        if waffle.switch_is_active('is_rkn'):
            for item in data:
                item['contacts'] = [i for i in item['contacts'] if i['type'] in RKN_CONTACT_TYPES]

        self._normalize_account_ids(data)
        self._set_is_available(data)
        return data
