import logging
import threading

from intranet.femida.src.monitoring.utils import alarm


logger = logging.getLogger(__name__)


class IrreversibleCheckpointCollection:
    """
    Коллекция с данными о необратимых изменениях.

    Основная идея: мониторить наличие необратимых изменений,
    используя для этого готовый механизм celery-results.
    Когда это нужно? Например, мы перевели тикет в Трекере в другой статус,
    после чего поймали ошибку. В итоге, изменения в БД мы откатим
    и получим состояние, при котором нельзя повторно выполнить действие,
    т.к. тикет уже переведен в другой статус.
    На примере выше, мы можем, используя данную коллекцию,
    пометить перевод статуса тикета, как необратимое изменение,
    и в случае последующих ошибок заранее узнать о том,
    что у нас есть неконсистентное состояние, которое блокирует процесс.

    Используется как singleton (irreversible_checkpoints).
    Есть одна общая коллекция (объект данного класса),
    с которой и работаем везде. Работает это так:
    1. После важных необратимых изменений создаем контрольную точку (checkpoint),
       задаем название и сохраняем данные важные для анализа
       потенциальной проблемы. Пример:
       ...
       irreversible_checkpoints.add(
           name='checkpoint_name',
           offer_id=1,
           user_id=2,
       )
       ...
    2. В случае ошибки закрываем коллекцию.
       Если в коллекции были контрольные точки,
       все это попадет в celery-results для дальнейшего разбора.
       ...
       irreversible_checkpoints.close()
       ...

    Для всех запросов работает автоматика на уровне middleware и exception_handler'а.
    Т.е. коллекция автоматически закрывается при ошибках во время запроса,
    и все контрольные точки попадают в celery-results.
    """
    def __init__(self):
        self._data = threading.local()

    def init(self):
        self._data.checkpoints = []

    def add(self, name, **data):
        self._data.checkpoints.append({
            'name': name,
            'data': data,
        })

    def close(self):
        checkpoints = getattr(self._data, 'checkpoints', [])
        if checkpoints:
            alarm('Irreversible checkpoints', checkpoints)
            self.init()


irreversible_checkpoints = IrreversibleCheckpointCollection()
