import itertools
import os

from django.utils import translation
from django.utils.functional import cached_property

from intranet.femida.src.core.choices import LANGUAGES
from intranet.femida.src.notifications import notify
from intranet.femida.src.notifications.utils import get_base_context
from intranet.femida.src.utils.translation import get_language
from intranet.femida.src.actionlog.models import actionlog
from . import serializers


class ReceiversFetcher:
    """
    Класс для удобного объявления получателей нотификаций.
    Используется как вспомогательный для нотификаций.
    Может быть использован только в качестве
    переменной объекта класса FetchingNotificationBase.

    Пример использования:

    >>> from collections import namedtuple
    >>>
    >>> Instance = namedtuple('Instance', ('created_by', 'modified_by'))
    >>> instance = Instance('creator@email.com', 'modifier@email.com')
    >>> initiator = 'creator@email.com'
    >>>
    >>> created_by_rcv = ReceiversFetcher(lambda x: [x.instance.created_by])
    >>> modified_by_rcv = ReceiversFetcher(lambda x: [x.instance.modified_by])
    >>> initiator_rcv = ReceiversFetcher(lambda x: [x.initiator])
    >>>
    >>> class Notification(FetchingNotificationBase):
    >>>     receivers = created_by_rcv + modified_by_rcv - initiator_rcv
    >>>
    >>> notification = Notification(instance, initiator)
    >>> print(notification.receivers.fetch())
    >>> # {'modifier@email.com'}

    """
    def __init__(self, fetchers=None, invert_fetchers=None):
        fetchers = fetchers or []
        invert_fetchers = invert_fetchers or []

        if not isinstance(fetchers, (list, tuple)):
            fetchers = [fetchers]
        if not isinstance(invert_fetchers, (list, tuple)):
            invert_fetchers = [invert_fetchers]

        self.fetchers = fetchers
        self.invert_fetchers = invert_fetchers

    def __get__(self, instance, owner):
        assert isinstance(instance, FetchingNotificationBase)
        self.notification = instance
        return self

    def __set__(self, instance, value):
        """
        Note: при присваивании происходит фактическое копирование с перезаписью,
        а не подмена ссылок.
        """
        assert isinstance(value, self.__class__)
        assert isinstance(instance, FetchingNotificationBase)
        self.notification = instance
        self.fetchers = value.fetchers
        self.invert_fetchers = value.invert_fetchers

    def __add__(self, other):
        assert isinstance(other, self.__class__)
        return self.__class__(
            fetchers=self.fetchers + other.fetchers,
            invert_fetchers=self.invert_fetchers + other.invert_fetchers,
        )

    def __sub__(self, other):
        assert isinstance(other, self.__class__)
        return self.__class__(
            fetchers=self.fetchers + other.invert_fetchers,
            invert_fetchers=self.invert_fetchers + other.fetchers,
        )

    def __iter__(self):
        return iter(self.fetch())

    @property
    def _fetched(self):
        return [f(self.notification) for f in self.fetchers]

    @property
    def _invert_fetched(self):
        return [f(self.notification) for f in self.invert_fetchers]

    def fetch(self):
        return (
            set(itertools.chain.from_iterable(self._fetched))
            - set(itertools.chain.from_iterable(self._invert_fetched))
        )


R = ReceiversFetcher


class NotificationBase:
    """
    Класс уведомления: формирует и отправляет письмо.

    Формируются заголовки письма.
    Шаблон письма связывается с объектом instance для дальнейшей параметризации.
    """
    subject = None
    template_name = None
    is_thread_beginning = False
    event_type = None
    supported_languages = [LANGUAGES.ru]

    def __init__(self, instance, initiator=None, headers=None, **kwargs):
        self.instance = instance
        self.initiator = initiator
        self.headers = headers or {}
        self.kwargs = kwargs

    @cached_property
    def necessary_languages(self):
        return self.get_necessary_languages()

    def get_necessary_languages(self):
        return self.supported_languages

    def get_subject(self, *args, **kwargs):
        return self.subject

    def get_subject_by_lang(self):
        subjects = {}
        for lang in self.necessary_languages:
            subjects[lang] = getattr(self, 'get_subject_' + lang, self.get_subject)()
        return subjects

    def get_cc(self):
        return []

    def get_thread_id(self):
        return None

    def get_headers(self):
        raise NotImplementedError

    def get_template_name(self):
        return self.template_name

    def localize_template_name(self, lang=LANGUAGES.ru):
        if lang == LANGUAGES.ru:
            return self.template_name
        name, ext = os.path.splitext(self.template_name)
        return f'{name}-{lang}{ext}'

    def get_template_name_by_lang(self):
        return {lang: self.localize_template_name(lang) for lang in self.necessary_languages}

    @cached_property
    def common_context(self):
        """
        Контекст шаблона отбивки, общий для всех получателей
        """
        return self.get_common_context()

    def get_common_context(self):
        context = get_base_context()
        context['instance'] = self.instance
        context['initiator'] = serializers.UserSerializer(self.initiator).data
        context.update(self.kwargs)
        return context

    def get_context_by_lang(self):
        context = {}
        user_lang = get_language()
        for lang in self.necessary_languages:
            translation.activate(lang)
            context[lang] = self.get_common_context()
        translation.activate(user_lang)
        return context

    def send(self, **kwargs):
        raise NotImplementedError

    def _set_default_params(self, kwargs):
        kwargs.setdefault('initiator', self.initiator)
        kwargs.setdefault('reply_to', self.initiator)

        if self.is_thread_beginning:
            kwargs.setdefault('message_id', self.get_thread_id())
            kwargs.setdefault('in_reply_to', None)
        else:
            kwargs.setdefault('message_id', None)
            kwargs.setdefault('in_reply_to', self.get_thread_id())

    def get_event_type(self):
        return (
            self.event_type
            or (actionlog.ctx.action_name if actionlog.is_initialized() else '')
        )

    def get_default_femida_headers(self):
        return {
            'X-Femida-From': self.initiator.username if self.initiator else '',
            'X-Femida-Event-Type': self.get_event_type() or '',
        }

    def get_femida_headers(self):
        return self.get_default_femida_headers()

    def __call__(self, **kwargs):
        self.send(**kwargs)


class FetchingNotificationBase(NotificationBase):
    """
    Класс уведомления, определяющий список получателей через ReceiversFetcher.

    Для определения списка получателей при описании подкласса перегрузить атрибут receivers
    экземляром класса ReceiversFetcher.
    Получатели будут вычислены при отправке нотификации.
    """
    receivers = R()
    transport = 'email'

    @cached_property
    def fetched_receivers(self):
        return self.fetch_receivers()

    def fetch_receivers(self):
        return self.receivers.fetch()

    def get_necessary_languages(self):
        languages = {getattr(r, 'lang', LANGUAGES.ru) for r in self.fetched_receivers}
        necessary_languages = set()
        for lang in languages:
            if lang in self.supported_languages:
                necessary_languages.add(lang)
            else:
                necessary_languages.add(LANGUAGES.ru)
        return list(necessary_languages)

    def send(self, **kwargs):
        self._set_default_params(kwargs)

        template_names = self.get_template_name_by_lang()
        cc = self.get_cc()
        subjects = self.get_subject_by_lang()
        headers = self.get_default_femida_headers()
        headers.update(self.headers or self.get_femida_headers())
        context = self.get_context_by_lang()

        user_lang = get_language()
        for receiver in self.fetched_receivers:
            language = getattr(receiver, 'lang', LANGUAGES.ru)
            if language not in self.necessary_languages:
                language = LANGUAGES.ru
            receiver_context = context[language]
            receiver_context['receiver'] = receiver
            receiver_context['language'] = language
            translation.activate(language)
            notify_kwargs = {
                'transport': self.transport,
                'template_name': template_names[language],
                'context': receiver_context,
                'receiver': receiver,
            }
            if self.transport == 'email':
                email_kwargs = {
                    'subject': subjects[language],
                    'cc': cc,
                    'headers': headers,
                }
                notify_kwargs = notify_kwargs | email_kwargs | kwargs
            notify(**notify_kwargs)
        translation.activate(user_lang)


class PersonalizedNotificationBase(NotificationBase):
    """
    Класс уведомления, поддерживающий персонализированные по получателям
    контекст шаблона и заголовки.

    Получатели уведомления задаются как ключи словаря context_personalization.
    """
    def __init__(self, instance, initiator=None, **kwargs):
        super().__init__(instance, initiator, **kwargs)
        self.context_personalization = self.get_context_personalization()
        self.headers_personalization = self.get_headers_personalization()

    def get_subject(self, receiver):
        return super().get_subject()

    def send(self, **kwargs):
        self._set_default_params(kwargs)

        template_name = self.get_template_name()
        cc = self.get_cc()
        for receiver in self.receivers:
            notify(
                transport='email',
                template_name=template_name,
                context=self.receiver_context_map[receiver],
                subject=self.get_subject(receiver),
                receiver=receiver,
                cc=cc,
                headers=self.receiver_headers_map.get(receiver, {}),
                **kwargs
            )

    @property
    def receivers(self):
        """
        Получатели.
        Для нотификаций, которые наследуются от PersonalizedNotificationBase,
        получателями являются те, для кого задан песонализированный контекст.
        Иными словами, это ключи словаря self.receiver_context_map
        """
        return self.receiver_context_map.keys()

    @cached_property
    def receiver_context_map(self):
        """
        Словарь вида {получатель: контекст_с_персонализацией_для_получателя}
        """
        return {
            receiver: dict(
                self.common_context,
                receiver=receiver,
                **receiver_context
            )
            for receiver, receiver_context in self.context_personalization.items()
        }

    def get_context_personalization(self):
        """
        Словарь вида {получатель: поля_специфичные_для_получателя}
        """
        return {}

    @cached_property
    def receiver_headers_map(self):
        common_headers = self.get_femida_headers()
        return {
            receiver: dict(
                common_headers,
                **self.headers_personalization.get(receiver, {})
            )
            for receiver in self.receivers
        }

    def get_headers_personalization(self):
        return {}
