import logging
from collections import defaultdict

import waffle

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.mail import make_msgid
from django.db import models
from django.template import Template, Context
from django.template.loader import render_to_string

from intranet.femida.src.candidates.choices import VERIFICATION_TYPES
from intranet.femida.src.communications.choices import MESSAGE_TEMPLATE_TYPES
from intranet.femida.src.communications.models import MessageTemplate
from intranet.femida.src.interviews.choices import APPLICATION_STATUSES
from intranet.femida.src.notifications.base import (
    R,
    FetchingNotificationBase,
    PersonalizedNotificationBase,
)
from intranet.femida.src.notifications.headers import (
    get_application_headers,
    get_candidate_headers,
    get_vacancy_headers,
    get_vacancy_headers_prefetched,
)
from intranet.femida.src.notifications.utils import (
    get_candidate_message_id,
    get_hiring_managers,
    get_candidate_email_subject,
)
from intranet.femida.src.utils.pluralization import get_russian_plural_form
from intranet.femida.src.vacancies.models import VacancyMembership
from intranet.femida.src.vacancies.choices import (
    VACANCY_ROLES,
    VACANCY_STATUSES,
    VACANCY_PRO_LEVELS_TRANSLATIONS,
)
from intranet.femida.src.tasks import send_email
from . import serializers


logger = logging.getLogger(__name__)
User = get_user_model()


def _get_hiring_teams_members(notification):
    applications_in_progress = notification.instance.applications.filter(
        status=APPLICATION_STATUSES.in_progress,
    )
    memberships = (
        VacancyMembership.unsafe
        .filter(
            vacancy__applications__in=applications_in_progress,
            role__in=(
                VACANCY_ROLES.hiring_manager,
                VACANCY_ROLES.head,
                VACANCY_ROLES.main_recruiter,
                VACANCY_ROLES.recruiter,
                VACANCY_ROLES.responsible,
            ),
        )
        .select_related('member')
    )
    return {m.member for m in memberships}


def _get_applications_related_receivers(applications):
    receivers = set()
    for application in applications:
        receivers |= set(application.vacancy.recruiters)
        receivers |= set(application.vacancy.responsibles)
        receivers |= get_hiring_managers(application)
    return receivers


candidate_responsibles_rcv = R(lambda x: x.instance.responsibles.all())
initiator_rcv = R(lambda x: [x.initiator])
hiring_teams_members_rcv = R(_get_hiring_teams_members)
applications_related_rcv = R(lambda x: (
    _get_applications_related_receivers(x.applications)
))
verification_candidate_responsibles = R(lambda x: x.instance.candidate.responsibles.all())
verification_creator = R(lambda x: [x.instance.created_by])


class CandidateNotificationMixin:
    """
    Mixin для нотификаций с новой версткой
    """
    def get_thread_id(self):
        return get_candidate_message_id(self.instance)

    def get_common_context(self):
        context = super().get_common_context()
        context['candidate'] = serializers.CandidateSerializer(self.instance).data
        interviews = self.kwargs.get('interviews')
        if isinstance(interviews, models.QuerySet):
            interviews = interviews.select_related('interviewer')
        context['interviews'] = serializers.InterviewSerializer(interviews, many=True).data
        return context


class CandidateFetchingNotification(CandidateNotificationMixin, FetchingNotificationBase):
    subject_prefix = None

    def get_subject(self):
        return get_candidate_email_subject(self.instance, self.subject_prefix)

    def get_femida_headers(self):
        headers = super().get_femida_headers()
        candidate_headers = get_candidate_headers(self.instance.id)
        headers.update(candidate_headers)
        return headers


class CandidateCreatedNotification(CandidateFetchingNotification):

    subject_prefix = 'Кандидат добавлен'
    template_name = 'email/candidates/create.html'
    receivers = candidate_responsibles_rcv - initiator_rcv
    is_thread_beginning = True


class CandidateUpdatedNotification(CandidateFetchingNotification):

    subject_prefix = 'Кандидат изменен'
    template_name = 'email/candidates/update.html'
    receivers = candidate_responsibles_rcv - initiator_rcv


class CandidateCompletedNotification(CandidateFetchingNotification):

    subject_prefix = 'Готовы результаты секций'
    template_name = 'mail/candidates/complete.html'
    receivers = hiring_teams_members_rcv + candidate_responsibles_rcv

    def get_femida_headers(self):
        headers = super().get_femida_headers()
        application_ids = []
        vacancy_ids = []
        for interview in self.kwargs.get('interviews', []):
            if interview.application:
                application_ids.append(interview.application_id)
                vacancy_ids.append(interview.application.vacancy_id)
        headers.update(get_application_headers(application_ids))
        headers.update(get_vacancy_headers(vacancy_ids))
        return headers


class CandidateOpenedNotification(CandidateFetchingNotification):

    subject_prefix = 'Начата работа по кандидату'
    template_name = 'mail/candidates/open.html'
    receivers = candidate_responsibles_rcv - initiator_rcv


class CandidateClosedNotification(CandidateFetchingNotification):

    subject_prefix = 'Завершена работа по кандидату'
    template_name = 'mail/candidates/close.html'
    receivers = candidate_responsibles_rcv - initiator_rcv


class CandidateApplicationBulkCreatedNotification(CandidateNotificationMixin,
                                                  PersonalizedNotificationBase):

    template_name = 'mail/candidates/applications-bulk-create.html'
    event_type = 'application_create'

    def __init__(self, instance, initiator=None, **kwargs):
        self.applications_by_receivers = self._get_applications_grouped_by_receivers(
            applications=kwargs['applications'],
            initiator=initiator,
        )
        super().__init__(
            instance=instance,
            initiator=initiator,
            **kwargs
        )

    def get_subject(self, receiver):
        applications = self.receiver_context_map[receiver]['applications']
        count = len(applications)
        prefix = 'Рассматривается на {count} ваканси{plural_suffix}'.format(
            count=count,
            plural_suffix=get_russian_plural_form(count, 'ю', 'и', 'й'),
        )
        return get_candidate_email_subject(self.instance, prefix)

    def _get_applications_grouped_by_receivers(self, applications, initiator):
        """
        Список претендентств для каждого получателя
        """
        applications_by_receivers = defaultdict(list)
        for application in applications:
            # Рекрутеры
            receivers = set(application.vacancy.recruiters)
            receivers.update(application.candidate.responsibles.all())
            # Команда, в которую найм
            if application.status == APPLICATION_STATUSES.in_progress:
                receivers.update(
                    application.vacancy.responsibles,
                    get_hiring_managers(application),
                )

            receivers.discard(initiator)
            for receiver in receivers:
                applications_by_receivers[receiver].append(application)

        return applications_by_receivers

    def get_context_personalization(self):
        context_personalization = {}
        for receiver, applications in self.applications_by_receivers.items():
            serialized_applications = [
                {
                    'id': application.id,
                    'is_receiver_team_member': receiver in application.vacancy.team,
                    'vacancy': serializers.VacancySerializer(application.vacancy).data,
                }
                for application in applications
            ]
            serialized_applications.sort(
                key=lambda x: x['is_receiver_team_member'],
                reverse=True,
            )
            context_personalization[receiver] = {
                'applications': serialized_applications,
            }
        return context_personalization

    def get_femida_headers(self):
        headers = super().get_femida_headers()
        candidate_headers = get_candidate_headers(self.instance.id)
        headers.update(candidate_headers)
        return headers

    def get_headers_personalization(self):
        headers_personalization = {}
        for receiver, applications in self.applications_by_receivers.items():
            headers = get_application_headers(
                [application.id for application in applications],
            )

            vacancy_headers = get_vacancy_headers_prefetched(
                [application.vacancy for application in applications]
            )
            headers.update(vacancy_headers)

            headers_personalization[receiver] = headers
        return headers_personalization


class CandidateProposedNotification(CandidateFetchingNotification):

    template_name = 'mail/candidates/proposals-create.html'
    event_type = 'application_create'
    receivers = (
        candidate_responsibles_rcv
        + applications_related_rcv
        - initiator_rcv
    )

    def __init__(self, instance, initiator, applications, pro_level_min, pro_level_max, **kwargs):
        super().__init__(
            instance=instance,
            initiator=initiator,
            applications=applications,
            pro_level_min=pro_level_min,
            pro_level_max=pro_level_max,
            **kwargs
        )
        self.applications = applications
        self.pro_level_min = pro_level_min
        self.pro_level_max = pro_level_max

    def get_subject(self):
        return (
            '[Предложение кандидата] {candidate_name} ({level})'.format(
                candidate_name=self.instance.get_full_name(),
                level=(
                    '{pro_level_min} - {pro_level_max}'
                    if self.pro_level_min is not None
                    else '{pro_level_max}'
                ).format(
                    pro_level_min=VACANCY_PRO_LEVELS_TRANSLATIONS.get(self.pro_level_min),
                    pro_level_max=VACANCY_PRO_LEVELS_TRANSLATIONS.get(self.pro_level_max),
                ),
            )
        )

    def get_femida_headers(self):
        headers = super().get_femida_headers()

        application_headers = get_application_headers(
            [application.id for application in self.applications],
        )
        headers.update(application_headers)

        vacancy_headers = get_vacancy_headers(
            [application.vacancy_id for application in self.applications],
        )
        headers.update(vacancy_headers)
        return headers


class SubmissionSeoCheckNotification(FetchingNotificationBase):

    template_name = 'email/seo-check.html'

    def get_subject(self):
        return '[SEO проверка] {first_name} {last_name}'.format(
            first_name=self.instance.first_name,
            last_name=self.instance.last_name,
        )

    def fetch_receivers(self):
        return ['job-seo-check@yandex-team.ru']

    def get_common_context(self):
        context = super().get_common_context()
        context['probability'] = self.kwargs.get('probability')
        context['vacancies'] = (
            self.instance.form.vacancies
            .filter(status=VACANCY_STATUSES.in_progress)
            .select_related('profession')
            .prefetch_related('memberships__member')
        )
        return context

    def send(self, **kwargs):
        kwargs['attachment_ids'] = [self.instance.attachment_id]
        super().send(**kwargs)


class VerificationNotificationBase(FetchingNotificationBase):

    def get_thread_id(self):
        return get_candidate_message_id(self.instance.candidate)

    def get_subject(self):
        return get_candidate_email_subject(self.instance.candidate, self.subject_prefix)

    def get_common_context(self):
        context = super().get_common_context()
        context['candidate'] = serializers.CandidateSerializer(self.instance.candidate).data
        return context

    def get_femida_headers(self):
        headers = self.get_default_femida_headers()
        candidate_headers = get_candidate_headers(self.instance.candidate.id)
        headers.update(candidate_headers)
        return headers


class VerificationWarningNotification(VerificationNotificationBase):

    subject_prefix = 'Ошибка анкеты КИ'
    template_name = 'email/verifications/warning.html'
    receivers = verification_creator
    event_type = 'candidate_verification_warning'


class VerificationSubmittedNotification(VerificationNotificationBase):

    subject_prefix = 'Анкета КИ заполнена'
    template_name = 'email/verifications/submitted_default.html'
    receivers = verification_candidate_responsibles + verification_creator
    event_type = 'candidate_verification_submission'

    def __init__(self, instance, **kwargs):
        super().__init__(instance, **kwargs)

        if instance.type == VERIFICATION_TYPES.default:
            self.template_name = 'email/verifications/submitted_default.html'
        elif instance.type == VERIFICATION_TYPES.international_by_grade:
            self.template_name = 'email/verifications/submitted_international_by_grade.html'


class VerificationSuccessNotification(VerificationNotificationBase):

    subject_prefix = 'Проверка КИ завершена - ОК'
    template_name = 'email/verifications/success.html'
    receivers = verification_candidate_responsibles + verification_creator
    event_type = 'candidate_verification_success'


class VerificationFailureNotification(VerificationNotificationBase):

    subject_prefix = 'Проверка КИ завершена - не ОК'
    template_name = 'email/verifications/failure.html'
    receivers = verification_candidate_responsibles + verification_creator
    event_type = 'candidate_verification_failure'


class VerificationMaybeDuplicateNotification(VerificationNotificationBase):
    subject_prefix = 'Мы нашли копию'
    template_name = 'email/verifications/maybe-duplicate.html'
    receivers = verification_creator
    event_type = 'candidate_verification_potential_duplicate'


def notify_all_interviews_completed(interview, initiator, interviews):
    canonical_interviews = [i for i in interviews if i.is_gradable]
    if len(canonical_interviews):
        notification = CandidateCompletedNotification(
            instance=interview.candidate,
            initiator=initiator,
            interviews=canonical_interviews,
        )
        notification.send()


def send_verification_form_to_candidate(verification_id, subject, text, receiver, sender):
    message_id = make_msgid()
    logger.info(
        'Sending form for Verification `%d` with Message-ID %s',
        verification_id,
        message_id,
    )
    body = render_to_string(
        template_name='email/verifications/send.html',
        context={'text': text},
    )
    send_email.delay(
        subject=subject,
        body=body,
        to=[receiver],
        reply_to=sender,
        from_email=sender,
        message_id=message_id,
        is_external=True,
    )


def send_verification_data_to_vendor(verification):
    message_id = make_msgid()
    logger.info(
        'Sending data from Verification `%d` with Message-ID %s to ESS Vendor for check',
        verification.id,
        message_id,
    )
    body = render_to_string(
        template_name='email/verifications/send-to-vendor.html',
        context={
            'verification': verification,
            'forms_host': settings.FORMS_EXT_HOST,
            'survey_id': settings.VERIFICATION_RESULT_SURVEY_ID,
            'is_result_link_enabled': waffle.switch_is_active('enable_verification_result_link'),
        },
    )
    send_email.delay(
        subject='Кандидат - %s' % verification.candidate.get_full_name(),
        body=body,
        to=['yandex@iris-russia.com'],
        reply_to='rec-org@yandex-team.ru',
        from_email='rec-org@yandex-team.ru',
        message_id=message_id,
        is_external=True,
    )


def send_certification_to_candidate(certification, receiver: str):
    message_id = make_msgid()
    logger.info(
        'Sending link to Candidate %s for Certification %s with Message-ID %s',
        certification.consideration.candidate_id,
        certification.id,
        message_id,
    )
    context = {
        'instance': certification,
    }

    if waffle.switch_is_active('enable_certification_new_template'):
        # Берём самый последний шаблон с типом certification
        template = (
            MessageTemplate.objects
            .filter(type=MESSAGE_TEMPLATE_TYPES.certification)
            .order_by('id')
            .last()
        )
        context = Context(context)
        body = Template(template.text).render(context)
    else:
        body = render_to_string(
            template_name='mail/certifications/send-to-candidate.html',
            context=context,
        )

    subject = 'Диагностика навыков — {}'.format(
        certification.consideration.candidate.get_full_name(),
    )
    send_email.delay(
        subject=subject,
        body=body,
        to=[receiver],
        reply_to=settings.NOREPLY_FROM_EMAIL,
        from_email=f'Яндекс<{settings.NOREPLY_FROM_EMAIL}>',
        message_id=message_id,
        is_external=True,
    )
