from django.db.models import prefetch_related_objects

from intranet.femida.src.candidates.models import CandidateCity, CandidateProfession, CandidateSkill
from intranet.femida.src.vacancies.models import VacancyCity, VacancySkill, Vacancy


def _get_header_string(values):
    if not values:
        return ''
    return '|{}|'.format(
        '|'.join(str(value) for value in values)
    )


def get_application_headers(application_ids):
    return {
        'X-Femida-Application-Ids': _get_header_string(application_ids),
    }


def get_candidate_headers(candidate_id):
    return {
        'X-Femida-Candidate-Id': candidate_id,
        'X-Femida-Candidate-City-Ids': _get_header_string(
            CandidateCity.objects
            .filter(candidate_id=candidate_id)
            .values_list('city_id', flat=True)
        ),
        'X-Femida-Candidate-Profession-Ids': _get_header_string(
            CandidateProfession.objects
            .filter(candidate_id=candidate_id)
            .values_list('profession_id', flat=True)
        ),
        'X-Femida-Candidate-Skill-Ids': _get_header_string(
            CandidateSkill.objects
            .filter(candidate_id=candidate_id)
            .values_list('skill_id', flat=True)
        ),
    }


def get_vacancy_headers(vacancy_ids):
    return {
        'X-Femida-Vacancy-Ids': _get_header_string(vacancy_ids),
        'X-Femida-Vacancy-City-Ids': _get_header_string(
            VacancyCity.objects
            .filter(vacancy__in=vacancy_ids)
            .values_list('city_id', flat=True)
            .distinct('city')
        ),
        'X-Femida-Vacancy-Profession-Ids': _get_header_string(
            Vacancy.unsafe
            .filter(
                id__in=vacancy_ids,
                profession__isnull=False,
            )
            .values_list('profession_id', flat=True)
            .distinct('profession')
        ),
        'X-Femida-Vacancy-Skill-Ids': _get_header_string(
            VacancySkill.objects
            .filter(vacancy__in=vacancy_ids)
            .values_list('skill_id', flat=True)
            .distinct('skill')
        ),
    }


def prefetch_candidate_data_for_headers(candidates):
    return prefetch_related_objects(
        candidates,
        'target_cities',
        'candidate_professions',
        'skills',
    )


def get_candidate_headers_prefetched(candidate):
    """
    Версия без запросов в базу при наличии ORM-кэша
    """
    city_ids = [c.id for c in candidate.target_cities.all()]
    profession_ids = [p.profession_id for p in candidate.candidate_professions.all()]
    skill_ids = [s.id for s in candidate.skills.all()]
    return {
        'X-Femida-Candidate-Id': candidate.id,
        'X-Femida-Candidate-City-Ids': _get_header_string(city_ids),
        'X-Femida-Candidate-Profession-Ids': _get_header_string(profession_ids),
        'X-Femida-Candidate-Skill-Ids': _get_header_string(skill_ids),
    }


def prefetch_vacancy_data_for_headers(vacancies):
    return prefetch_related_objects(
        vacancies,
        'cities',
        'skills',
    )


def get_vacancy_headers_prefetched(vacancies):
    """
    Версия без запросов в базу при наличии ORM-кэша
    """
    headers = {
        'X-Femida-Vacancy-Ids': set(),
        'X-Femida-Vacancy-City-Ids': set(),
        'X-Femida-Vacancy-Profession-Ids': set(),
        'X-Femida-Vacancy-Skill-Ids': set(),
    }
    for vacancy in vacancies:
        headers['X-Femida-Vacancy-Ids'].add(vacancy.id)
        headers['X-Femida-Vacancy-City-Ids'].update([city.id for city in vacancy.cities.all()])
        headers['X-Femida-Vacancy-Profession-Ids'].add(vacancy.profession_id)
        headers['X-Femida-Vacancy-Skill-Ids'].update([skill.id for skill in vacancy.skills.all()])

    for key, value in headers.items():
        headers[key] = _get_header_string(value)
    return headers
