import logging
import re
import waffle

from typing import Iterable, List

from constance import config
from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.mail.message import make_msgid
from django.template import loader
from django.template.loader import render_to_string
from django.utils.safestring import mark_safe

from intranet.femida.src.attachments.models import Attachment
from intranet.femida.src.candidates.helpers import get_main_email
from intranet.femida.src.notifications.base import FetchingNotificationBase, R
from intranet.femida.src.notifications.headers import get_vacancy_headers
from intranet.femida.src.notifications.utils import get_base_context, get_vacancy_message_id
from intranet.femida.src.offers.choices import DOCUMENT_TYPES, DOCUMENT_TYPES_TRANSLATIONS
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.startrek.operations import IssueCommentOperation
from intranet.femida.src.tasks import send_email


logger = logging.getLogger(__name__)

User = get_user_model()


def get_offer_email_subject(offer, prefix):
    return '[{prefix}] {full_name}'.format(
        prefix=prefix,
        full_name=offer.candidate.get_full_name(),
    )


def _get_hiring_managers(notification):
    if notification.instance.vacancy.hiring_manager:
        return [notification.instance.vacancy.hiring_manager]
    return []


vacancy_recruiters_rcv = R(lambda x: x.instance.vacancy.recruiters)
candidate_responsibles_rcv = R(lambda x: x.instance.candidate.responsibles.all())
initiator_rcv = R(lambda x: [x.initiator] if x.initiator else [])
hiring_manager_rcv = R(_get_hiring_managers)
creator_rcv = R(lambda x: [x.instance.creator])
employee_rcv = R(lambda x: [x.instance.employee])
bosses_rcv = R(lambda x: filter(None, [x.instance.current_boss, x.instance.future_boss]))
hr_partners_rcv = R(lambda x: x.instance.all_hr_partners)


class OfferNotification(FetchingNotificationBase):

    subject_prefix = None
    receivers = (
        vacancy_recruiters_rcv
        + candidate_responsibles_rcv
        + hiring_manager_rcv
        - initiator_rcv
    )

    def get_subject(self):
        return get_offer_email_subject(self.instance, self.subject_prefix)

    def get_thread_id(self):
        return get_vacancy_message_id(self.instance.vacancy)

    def get_femida_headers(self):
        headers = self.get_default_femida_headers()
        headers.update(get_vacancy_headers([self.instance.vacancy_id]))
        return headers


class OfferSentForApprovalNotification(OfferNotification):

    subject_prefix = 'Оффер отправлен на согласование'
    template_name = 'email/offers/approved.html'


class OfferSentNotification(OfferNotification):

    subject_prefix = 'Оффер отправлен'
    template_name = 'email/offers/sent.html'


class OfferAcceptedNotification(OfferNotification):

    subject_prefix = 'Оффер принят'
    template_name = 'email/offers/accepted.html'

    def send(self, **kwargs):
        super().send(
            reply_to=self.instance.creator,
            in_reply_to=None,
        )


class OfferInternalAcceptedNotification(OfferNotification):

    subject_prefix = 'Ротация согласована'
    template_name = 'email/offers/accepted-internal.html'
    receivers = employee_rcv
    cc_receivers = (
        creator_rcv
        + bosses_rcv
        + hr_partners_rcv
        - receivers
    )

    def send(self, **kwargs):
        super().send(
            reply_to=self.instance.creator,
            in_reply_to=None,
        )

    def get_subject(self):
        subject = super().get_subject()
        return '{} {}@'.format(subject, self.instance.employee.username)

    def get_cc(self):
        return [user.email for user in self.cc_receivers.fetch()]


class OfferDeletedNotification(OfferNotification):

    subject_prefix = 'Оффер удален'
    template_name = 'email/offers/deleted.html'
    event_type = 'offer_delete'


class OfferRejectedNotification(OfferDeletedNotification):

    subject_prefix = 'Оффер отозван'

    def get_common_context(self):
        context = super().get_common_context()
        context['reason'] = 'rejected'
        return context


# Опросы по найму


class OfferHRSurveyRecruiterNotification(FetchingNotificationBase):

    subject = 'Взаимодействие с заказчиком по вакансии'
    template_name = 'email/offers/hr-survey-recruiter.html'
    receivers = creator_rcv

    def get_cc(self):
        return [
            'adapt-notify@yandex-team.ru',
            'hr-poll@yandex-team.ru',
        ]


class OfferHRSurveyHiringManagerNotification(FetchingNotificationBase):

    subject = 'Оцените взаимодействие с рекрутером'
    template_name = 'email/offers/hr-survey-hiring-manager.html'
    receivers = hiring_manager_rcv

    def get_cc(self):
        return [
            'adapt-notify@yandex-team.ru',
            'hr-poll@yandex-team.ru',
        ]


# Внешняя коммуникация


def send_offer_to_candidate(offer_id: int, subject: str, message: str, offer_text: str,
                            receiver: str, sender: str, attachments: Iterable[Attachment] = None,
                            bcc: Iterable[User] = None) -> None:

    attachments = attachments or []
    attachment_ids = [a.id for a in attachments]

    bcc = bcc or []
    bcc = [u.email for u in bcc]

    # TODO: никогда не переделывать это на нормальный attachment в будущем

    message = message.replace('\n', '\n<br/>')
    offer_text = offer_text.replace('\n', '\n<br/>')

    url_simple_rgx = re.compile(r'(?P<url>https?://[a-zA-Z0-9\-/.=?&]+)')
    offer_text = url_simple_rgx.sub(r'<a href="\g<url>">\g<url></a>', offer_text)

    body = render_to_string('email/offers/send.html', {
        'message': mark_safe(message),
        'offer_text': mark_safe(offer_text),
    })

    message_id = make_msgid()
    logger.info('Sending offer `%d` with Message-ID %s', offer_id, message_id)

    send_email.delay(
        subject=subject,
        body=body,
        to=[receiver],
        reply_to=sender,
        from_email=sender,
        attachment_ids=attachment_ids,
        bcc=bcc,
        message_id=message_id,
        is_external=True,
    )


def send_docs_request_message_to_candidate(offer: Offer, link_url: str,
                                           errors: List[str] = None) -> None:
    if not waffle.switch_is_active('enable_detailed_document_requests'):
        errors = None

    body = render_to_string(
        template_name='email/offers/docs-request.html',
        context={
            'offer': offer,
            'fill_form_url': link_url,
            'errors': errors,
            'docs_request_timeout_hours': config.DOCS_REQUEST_TIMEOUT_HOURS,
        },
    )
    message_id = make_msgid()

    logger.info('Sending documents request `%d` with Message-ID %s', offer.id, message_id)
    send_email.delay(
        subject='От компании Яндекс',
        body=body,
        to=[get_main_email(offer.candidate)],
        cc=[offer.creator.email],
        from_email=settings.JOB_EMAIL_VERBOSE,
        message_id=message_id,
        is_external=True,
    )


def notify_hr_about_docs_processing_problem(offer: Offer) -> None:
    context = get_base_context()
    document_type = (
        DOCUMENT_TYPES_TRANSLATIONS.get(
            offer.passport_data['document_type'],
            DOCUMENT_TYPES_TRANSLATIONS[DOCUMENT_TYPES.other],
        )
        if offer.passport_data else None
    )
    context.update({
        'instance': offer,
        'passport_data': offer.passport_data,
        'document_type': document_type,
        'snils_number': offer.snils_number,
        'residence_address_data': offer.residence_address_data,
        'registration_address_data': offer.registration_address_data,
        'docs_request_timeout_hours': config.DOCS_REQUEST_TIMEOUT_HOURS,
    })

    operation = IssueCommentOperation(offer.startrek_hr_key)
    operation.delay(
        text=loader.render_to_string(
            template_name='startrek/offers/docs-processing-problem.wiki',
            context=context,
        ),
    )
