from rest_framework import serializers

from django.contrib.auth import get_user_model

from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.communications.models import Message
from intranet.femida.src.interviews.choices import (
    APPLICATION_STATUSES_TRANSLATIONS,
    APPLICATION_RESOLUTIONS_TRANSLATIONS,
)
from intranet.femida.src.interviews.models import Application, Interview
from intranet.femida.src.vacancies.choices import (
    VACANCY_STATUSES_TRANSLATIONS,
    VACANCY_PRO_LEVELS_TRANSLATIONS,
)
from intranet.femida.src.vacancies.models import Vacancy, PublicationSubscription
from intranet.femida.src.core.serializers import MapperField


User = get_user_model()


class UserSerializer(serializers.ModelSerializer):

    full_name = serializers.ReadOnlyField(source='get_full_name')

    class Meta:
        model = User
        fields = (
            'id',
            'username',
            'full_name',
            'gender',
            'timezone',
        )


class MessageSerializer(serializers.ModelSerializer):

    author = UserSerializer()

    class Meta:
        model = Message
        fields = (
            'id',
            'text',
            'html',
            'type',
            'status',
            'error',
            'author',
        )


class CandidateSerializer(serializers.ModelSerializer):

    full_name = serializers.ReadOnlyField(source='get_full_name')
    target_cities = serializers.SerializerMethodField()
    professions = serializers.SerializerMethodField()
    skills = serializers.SerializerMethodField()
    responsibles = UserSerializer(many=True)

    def get_target_cities(self, obj):
        return [i.name_ru for i in obj.target_cities.all()]

    def get_skills(self, obj):
        return [i.skill.name for i in obj.candidate_skills.all()]

    def get_professions(self, obj):
        return [i.profession.name for i in obj.candidate_professions.all()]

    class Meta:
        model = Candidate
        fields = (
            'id',
            'first_name',
            'last_name',
            'full_name',
            'gender',
            'city',
            'target_cities',
            'professions',
            'skills',
            'responsibles',
        )


class VacancySerializer(serializers.ModelSerializer):

    status_display = MapperField(VACANCY_STATUSES_TRANSLATIONS, source='status')
    pro_level_min_display = MapperField(VACANCY_PRO_LEVELS_TRANSLATIONS, source='pro_level_min')
    pro_level_max_display = MapperField(VACANCY_PRO_LEVELS_TRANSLATIONS, source='pro_level_max')
    profession = serializers.ReadOnlyField(source='profession.name')
    cities = serializers.SerializerMethodField()
    skills = serializers.SerializerMethodField()

    def get_cities(self, obj):
        return [i.name_ru for i in obj.cities.all()]

    def get_skills(self, obj):
        return [i.name for i in obj.skills.all()]

    class Meta:
        model = Vacancy
        fields = (
            'id',
            'name',
            'status',
            'status_display',
            'cities',
            'profession',
            'pro_level_min_display',
            'pro_level_max_display',
            'skills',
            'publication_title',
        )


class PublicationSubscriptionSerializer(serializers.ModelSerializer):

    pro_level_min_display = MapperField(VACANCY_PRO_LEVELS_TRANSLATIONS, source='pro_level_min')
    pro_level_max_display = MapperField(VACANCY_PRO_LEVELS_TRANSLATIONS, source='pro_level_max')
    department = serializers.CharField(source='department.name', default=None)
    professions = serializers.SerializerMethodField()
    cities = serializers.SerializerMethodField()
    skills = serializers.SerializerMethodField()
    abc_services = serializers.SerializerMethodField()
    include_inactive = serializers.SerializerMethodField()

    def get_cities(self, obj):
        return [i.name_ru for i in obj.cities.all()]

    def get_professions(self, obj):
        return [i.name for i in obj.professions.all()]

    def get_skills(self, obj):
        return [i.name for i in obj.skills.all()]

    def get_abc_services(self, obj):
        return [i.name for i in obj.abc_services.all()]

    def get_include_inactive(self, obj):
        return not obj.only_active

    class Meta:
        model = PublicationSubscription
        fields = (
            'text',
            'external_url',
            'department',
            'pro_level_min_display',
            'pro_level_max_display',
            'professions',
            'cities',
            'skills',
            'abc_services',
            'include_inactive',
        )


class InterviewSerializer(serializers.ModelSerializer):

    grade_display = serializers.ReadOnlyField(source='get_grade_display')
    interviewer = UserSerializer()

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'section',
            'grade',
            'grade_display',
            'is_hire',
            'interviewer',
            'finished',
            'comment',
            'formatted_comment',
            'event_url',
        )


class ApplicationSerializer(serializers.ModelSerializer):

    status_display = MapperField(APPLICATION_STATUSES_TRANSLATIONS, source='status')
    resolution_display = MapperField(APPLICATION_RESOLUTIONS_TRANSLATIONS, source='resolution')

    class Meta:
        model = Application
        fields = (
            'id',
            'status',
            'status_display',
            'proposal_status',
            'resolution_display',
        )
