import json
import logging

from collections import defaultdict, Counter
from datetime import timedelta, datetime

from django.conf import settings
from django.db.models import Count, Q
from django.utils import timezone

from intranet.femida.src.candidates.models import Candidate, CandidateSubmission
from intranet.femida.src.candidates.choices import (
    SUBMISSION_STATUSES,
    SUBMISSION_SOURCES,
    CANDIDATE_STATUSES,
)
from intranet.femida.src.celery_app import app
from intranet.femida.src.interviews.choices import INTERVIEW_UNFINISHED_STATES
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.notifications import notify
from intranet.femida.src.notifications.utils import get_base_context
from intranet.femida.src.staff.choices import DEPARTMENT_ROLES
from intranet.femida.src.users.models import User
from intranet.femida.src.utils.lock import locked_task
from intranet.femida.src.vacancies.choices import VACANCY_ROLES


logger = logging.getLogger(__name__)


@app.task
@locked_task
def remind_about_unfinished_interviews():
    today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)
    interviews = (
        Interview.unsafe
        .filter(
            state__in=INTERVIEW_UNFINISHED_STATES._db_values,
            event_id__isnull=False,
            event_start_time__gte=today - timedelta(days=2),
            event_start_time__lt=today,
        )
        .select_related('interviewer')
    )
    receiver_interviews_map = defaultdict(list)
    receivers = set()

    for interview in interviews:
        receivers.add(interview.interviewer)
        receiver_interviews_map[interview.interviewer.id].append(interview)

    context = get_base_context()
    for receiver in receivers:
        context['receiver'] = receiver
        context['interviews'] = receiver_interviews_map[receiver.id]
        notify(
            transport='email',
            template_name='email/interviews/unfinished-interviews.html',
            receiver=receiver,
            context=context,
            subject='[Напоминание] Завершите секции',
        )
    receiver_logins = [r.username for r in receivers]
    logger.info(
        '%d emails about unfinished interviews were sent. Receivers: %s',
        len(receiver_logins),
        ', '.join(receiver_logins),
    )


def _forgotten_candidates_filter(receiver: User, date_lte: datetime) -> str:
    filters = [
        {
            "field": "CandidateStatusFilter",
            "condition": "Equal",
            "is_active": True,
        },
        {
            "field": "LastConsiderationRecruiterFilter",
            "condition": "Any",
            "values": [receiver.username],
        },
        {
            "field": "LastConsiderationExtendedStatusChangedAtFilter",
            "condition": "Range",
            "extended_status_changed_at__gte": "",
            "extended_status_changed_at__lte": date_lte.strftime("%Y-%m-%d"),
        }
    ]
    return json.dumps(filters)


@app.task
@locked_task
def remind_about_forgotten_candidates():
    today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)
    four_weeks_ago = today - timedelta(weeks=4)
    ten_days_ago = today - timedelta(days=10)

    forgotten_candidates = (
        Candidate.unsafe
        .alive()
        .filter(
            status=CANDIDATE_STATUSES.in_progress,
            modified__lte=ten_days_ago,
            responsibles__isnull=False,
        )
        .distinct('id')
        .prefetch_related('responsibles')
    )

    receiver_candidate_map = defaultdict(list)
    for candidate in forgotten_candidates:
        for resp in candidate.responsibles.all():
            receiver_candidate_map[resp].append(candidate)

    for receiver, candidates in receiver_candidate_map.items():
        four_weeks_count = ten_days_count = 0
        for candidate in candidates:
            if candidate.modified <= four_weeks_ago:
                four_weeks_count += 1
            else:
                ten_days_count += 1
        context = dict(
            get_base_context(),
            receiver=receiver,
            ten_days_ago_filter=_forgotten_candidates_filter(receiver, ten_days_ago),
            ten_days_count=ten_days_count,
            four_weeks_ago_filter=_forgotten_candidates_filter(receiver, four_weeks_ago),
            four_weeks_count=four_weeks_count,
        )
        notify(
            transport='email',
            template_name='email/candidates/forgotten-candidates.html',
            receiver=receiver,
            context=context,
            subject='[Напоминание] Кандидаты без обновления',
        )
    receiver_logins = [r.username for r in receiver_candidate_map]
    logger.info(
        '%d emails about long candidates were sent. Receivers: %s',
        len(receiver_logins),
        ', '.join(receiver_logins),
    )


def _get_submissions_count_by_source(source):
    if source == SUBMISSION_SOURCES.publication:
        prefix = 'publication__vacancy__memberships__'
    else:
        prefix = f'{source}__vacancies__memberships__'

    is_recruiter_q = Q(**{
        prefix + 'role__in': (
            VACANCY_ROLES.recruiter,
            VACANCY_ROLES.main_recruiter,
        ),
    })
    return dict(
        CandidateSubmission.unsafe
        .filter(
            is_recruiter_q,
            status=SUBMISSION_STATUSES.new,
        )
        .values_list(prefix + 'member__username')
        .annotate(Count('id', distinct=True))
    )


@app.task
@locked_task
def remind_about_new_submissions():
    submissions_by_username = sum([
        Counter(_get_submissions_count_by_source(source))
        for source in (
            SUBMISSION_SOURCES.form,
            SUBMISSION_SOURCES.reference,
            SUBMISSION_SOURCES.rotation,
            SUBMISSION_SOURCES.publication,
        )
    ], Counter())

    for receiver, submissions_count in submissions_by_username.items():
        context = dict(
            get_base_context(),
            receiver=receiver,
            submissions_count=submissions_count,
        )
        notify(
            transport='email',
            template_name='email/submissions/new-submissions.html',
            receiver='{}@yandex-team.ru'.format(receiver),
            context=context,
            subject='[Напоминание] Новые отклики',
        )
    receiver_logins = submissions_by_username.keys()
    logger.info(
        '%d emails about new submissions were sent. Receivers: %s',
        len(receiver_logins),
        ', '.join(receiver_logins),
    )


@app.task
@locked_task
def complain_about_unfinished_interviews():
    department_id = settings.DEPARTMENT_SEARCH_AND_ADVERTISEMENTS_ID
    department_chief = (
        User.objects
        .filter(
            department_id=department_id,
            department_users__role=DEPARTMENT_ROLES.chief
        )
        .prefetch_related('subordinates')
        .first()
    )
    if not department_chief:
        logger.error(
            "Could not find a chief for department `{department_id}` "
            "to send complains about unfinished interviews"
        )
        return
    subordinates_rec = list(department_chief.subordinates_rec)
    chiefs = [
        subordinate
        for subordinate in subordinates_rec
        if subordinate.chief_id == department_chief.username
    ]
    chief_subordinates_map = _group_subordinates_by_chiefs(subordinates_rec, chiefs)

    interviewers_ids = [
        subordinate.pk
        for subordinates in chief_subordinates_map.values()
        for subordinate in subordinates
    ]
    interviews = _get_unfinished_interviews(interviewers_ids)
    receivers = set()
    receiver_interviews_map = defaultdict(list)
    for interview in interviews:
        receiver = interview.application.vacancy.main_recruiter.recruitment_partner
        if receiver:
            receivers.add(receiver)
            receiver_interviews_map[receiver.pk].append(interview)

    _send_all_unfinished_interview_complains(
        receivers,
        chiefs,
        receiver_interviews_map,
        chief_subordinates_map,
    )


def _get_unfinished_interviews(interviewers_ids):
    today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)
    return (
        Interview.unsafe
        .filter(
            state__in=INTERVIEW_UNFINISHED_STATES._db_values,
            event_id__isnull=False,
            event_start_time__lt=today,
            application__isnull=False,
            interviewer__pk__in=interviewers_ids,
        )
        .select_related('interviewer')
        .prefetch_related('application__vacancy__memberships__member')
    )


def _group_subordinates_by_chiefs(subordinates, chiefs):
    subordinate_by_username = {
        subordinate.username: subordinate
        for subordinate in subordinates
    }
    subordinate_by_id = {
        subordinate.pk: subordinate
        for subordinate in subordinates
    }
    subordinates_ids = set(subordinate.pk for subordinate in subordinates)
    chiefs_ids = set(chief.pk for chief in chiefs)
    chief_subordinates_map = defaultdict(list)
    subordinate_chain_chief_map = {}
    while subordinates_ids:
        subordinate_id = subordinates_ids.pop()
        if subordinate_id in subordinate_chain_chief_map:
            continue
        subordinate = subordinate_by_id[subordinate_id]
        subordinates_chain = []
        chain_chief = None
        while subordinate.pk not in chiefs_ids:
            subordinates_chain.append(subordinate)
            subordinate_chief = subordinate_by_username[subordinate.chief_id]
            if subordinate_chief.pk in subordinate_chain_chief_map:
                chain_chief = subordinate_chain_chief_map[subordinate_chief.pk]
                break
            if subordinate_chief.pk in subordinates_ids:
                subordinates_ids.remove(subordinate_chief.pk)
            subordinate = subordinate_chief
        if chain_chief is None:
            chain_chief = subordinate
        for sub in subordinates_chain:
            subordinate_chain_chief_map[sub.pk] = chain_chief
        chief_subordinates_map[chain_chief.pk].extend(subordinates_chain)
    return chief_subordinates_map


def _send_all_unfinished_interview_complains(
        receivers,
        chiefs,
        receiver_interviews_map,
        chief_subordinates_map,
):
    for receiver in receivers:
        interviewer_interviews_map = defaultdict(list)
        for interview in receiver_interviews_map[receiver.pk]:
            interviewer_interviews_map[interview.interviewer.pk].append(interview)
        complains = []
        for chief in chiefs:
            interviewers_with_unfinished_interviews = [
                interviewer
                for interviewer in chief_subordinates_map[chief.pk]
                if interviewer.pk in interviewer_interviews_map
            ]
            if not interviewers_with_unfinished_interviews:
                continue
            complains.append(
                _serialize_complain(
                    chief=chief,
                    interviewers=interviewers_with_unfinished_interviews,
                    interviewer_interviews_map=interviewer_interviews_map,
                )
            )
        if complains:
            _notify_about_unfinished_interview_complains(receiver, complains)
            logger.info(
                'Complains about unfinished interviews [%s] were sent to %s',
                ','.join([
                    str(interview['id'])
                    for complain in complains
                    for interviewer in complain['interviewers']
                    for interview in interviewer['interviews']
                ]),
                receiver.username,
            )


def _serialize_complain(chief, interviewers, interviewer_interviews_map):
    return {
        'chief': {
            'full_name': f'{chief.first_name} {chief.last_name}',
            'username': f'{chief.username}',
        },
        'interviewers': [
            {
                'full_name': f'{interviewer.first_name} {interviewer.last_name}',
                'username': f'{interviewer.username}',
                'interviews': [
                    {
                        'date': interview.event_start_time.strftime('%d.%m.%Y'),
                        'id': interview.id,
                        'section': interview.section,
                    }
                    for interview in interviewer_interviews_map[interviewer.pk]
                ],
            }
            for interviewer in interviewers
        ]
    }


def _notify_about_unfinished_interview_complains(receiver, complains):
    context = get_base_context()
    context['receiver'] = receiver
    context['complains'] = complains
    notify(
        transport='email',
        template_name='email/interviews/complain-unfinished-interviews.html',
        receiver=receiver,
        context=context,
        subject='Найм: нужен ответ по секциям',
    )
