from django.conf import settings
from django.utils import timezone


def get_base_context():
    return {
        'femida_url': settings.FEMIDA_URL,
        'staff_url': settings.STAFF_URL,
        'startrek_url': settings.STARTREK_URL,
        'abc_url': settings.ABC_URL,
        'contest_url': settings.CONTEST_URL,
        'contest_interview_url': settings.CONTEST_INTERVIEW_URL,
        'contest_host': settings.CONTEST_HOST,
        'ok_url': settings.OK_URL,
        'cab_url': settings.CAB_URL,
        'oebs_hire_url': settings.OEBS_HIRE_URL,
    }


def _get_message_id(prefix, instance_id, timestamp=None, host=None):
    timestamp = timestamp or timezone.now()
    host = host or settings.FEMIDA_HOST
    return '<{prefix}#{instance_id}-{timestamp}@{host}>'.format(
        prefix=prefix,
        instance_id=instance_id,
        timestamp=timestamp.strftime('%Y%m%d%H%M%S'),
        host=host,
    )


def get_communication_message_id(candidate):
    """
    MessageId, по которому мы будем восстанавливать претендентство при ответе кандидата

    FEMIDA-2572 Про `cand` см. комментарий в create_message_from_separator
    """
    return _get_message_id(
        'appl',
        'cand%d' % candidate.id,
        host='yandex-team.ru',
    )


def get_problem_message_id(problem):
    return _get_message_id('problem', problem.id, problem.created)


def get_candidate_message_id(candidate):
    return _get_message_id('cand', candidate.id, candidate.created)


def get_vacancy_message_id(vacancy):
    return _get_message_id('vac', vacancy.id, vacancy.created)


def get_hiring_managers(application):
    if application.vacancy.hiring_manager:
        return {application.vacancy.hiring_manager}
    return set()


def get_candidate_email_subject(candidate, prefix, suffix=''):
    suffix = f' — {suffix}' if suffix else ''
    return f'[{prefix}] {candidate.get_full_name()}{suffix}'


def get_application_email_subject(application, prefix):
    suffix = application.vacancy.composite_name
    return get_candidate_email_subject(application.candidate, prefix, suffix)


def get_interview_email_subject(interview, prefix):
    suffix = interview.application.vacancy.composite_name if interview.application else ''
    return get_candidate_email_subject(interview.candidate, prefix, suffix)


def get_problem_email_subject(problem, prefix):
    return f'[{prefix}] {problem.summary}'
