from django.contrib.auth import get_user_model

from intranet.femida.src.notifications.base import FetchingNotificationBase, R
from intranet.femida.src.notifications.headers import get_vacancy_headers
from intranet.femida.src.notifications.utils import get_vacancy_message_id
from intranet.femida.src.vacancies.choices import VACANCY_TYPES


User = get_user_model()


def get_vacancy_email_subject(vacancy, prefix):
    return '[{prefix}] {vacancy_name}'.format(
        prefix=prefix,
        vacancy_name=vacancy.composite_name,
    )


def _get_hiring_managers(notification):
    if notification.instance.hiring_manager:
        return [notification.instance.hiring_manager]
    return []


created_by_rcv = R(lambda x: (
    [x.instance.created_by] if x.instance.created_by else []
))
initiator_rcv = R(lambda x: [x.initiator] if x.initiator else [])
vacancy_main_recruiter_rcv = R(lambda x: (
    [x.instance.main_recruiter] if x.instance.main_recruiter else []
))
vacancy_researchers_rcv = R(lambda x: x.instance.researchers)
hiring_manager_rcv = R(_get_hiring_managers)


class VacancyNotification(FetchingNotificationBase):

    subject_prefix = None
    receivers = (
        vacancy_researchers_rcv
        + hiring_manager_rcv
        - initiator_rcv
    )

    def get_subject(self):
        return get_vacancy_email_subject(self.instance, self.subject_prefix)

    def get_thread_id(self):
        return get_vacancy_message_id(self.instance)

    def get_femida_headers(self):
        headers = self.get_default_femida_headers()
        headers.update(get_vacancy_headers([self.instance.id]))
        return headers


class VacancyCreatedNotification(VacancyNotification):

    subject_prefix = 'Новая вакансия'
    template_name = 'email/vacancies/create.html'
    receivers = created_by_rcv + hiring_manager_rcv
    is_thread_beginning = True

    def get_subject(self):
        subject_prefixes_map = {
            VACANCY_TYPES.new: self.subject_prefix,
            VACANCY_TYPES.replacement: 'Вакансия на замену',
            VACANCY_TYPES.internship: 'Вакансия на стажировку'
        }
        subject_prefix = subject_prefixes_map.get(self.instance.type, self.subject_prefix)
        return get_vacancy_email_subject(self.instance, subject_prefix)


class VacancyApprovedNotification(VacancyNotification):

    subject_prefix = 'Вакансия подтверждена'
    template_name = 'email/vacancies/approve.html'
    receivers = hiring_manager_rcv


class VacancyClosedNotification(VacancyNotification):

    subject_prefix = 'Вакансия закрыта'
    template_name = 'email/vacancies/close.html'


class VacancyResumedNotification(VacancyNotification):

    subject_prefix = 'Вакансия возобновлена'
    template_name = 'email/vacancies/resume.html'


class VacancySuspendedNotification(VacancyNotification):

    subject_prefix = 'Вакансия приостановлена'
    template_name = 'email/vacancies/suspend.html'


class VacancyUpdatedNotification(VacancyNotification):

    subject_prefix = 'Вакансия изменена'
    template_name = 'email/vacancies/update.html'
    receivers = (
        vacancy_main_recruiter_rcv
        + vacancy_researchers_rcv
        + hiring_manager_rcv
        - initiator_rcv
    )
